# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import os
import logging
from abc import ABCMeta, abstractmethod
from datetime import datetime, timedelta
from typing import Any, List, Optional, Union

import requests
try:
    from ticket_parser2_py3.api.v1 import BlackboxEnv
except ImportError:
    from ticket_parser2.api.v1 import BlackboxEnv
from travel.library.python.tvm_ticket_provider.clients import (
    AbstractTvmClient,
    QloudTvmClient,
    DeployTvmClient,
    TvmClient,
    TvmToolClient,
)
from travel.library.python.tvm_ticket_provider.models import ServiceTicket, UserTicket  # noqa

DEFAULT_TIMEOUT = 0.01


class TvmError(Exception):
    def __init__(self, message):
        self.message = message

    def __str__(self):
        return '<{} message={}/>'.format(
            self.__class__.__name__,
            self.message
        )


class TvmUnknownDestinationError(TvmError):
    pass


class TvmUpdateTicketError(TvmError):
    pass


class AbstractTvmTicketProvider(object):
    __metaclass__ = ABCMeta

    def __init__(self, client, destinations, logger, ticket_life_time=5):
        # type: (Optional[AbstractTvmClient], List[int], logging.Logger, int) -> None
        self._client = client
        self._destinations = destinations

        self._last_update_ticket_time = None
        self._ticket_by_destination = {}
        self._ticket_life_time = ticket_life_time

        self._logger = logger

    @abstractmethod
    def get_ticket(self, destination):
        # type: (Union[int, str]) -> str
        pass

    @abstractmethod
    def check_service_ticket(self, ticket):
        # type: (str) -> ServiceTicket
        pass

    @abstractmethod
    def check_user_ticket(self, ticket):
        # type: (str) -> UserTicket
        pass

    def _need_update_ticket(self):
        # type: () -> bool
        if not self._last_update_ticket_time:
            return True

        delta = datetime.now() - self._last_update_ticket_time
        if delta >= timedelta(minutes=self._ticket_life_time):
            return True

        return False


class FakeTvmTicketProvider(AbstractTvmTicketProvider):
    def get_ticket(self, destination):
        return None

    def check_service_ticket(self, ticket):
        return None

    def check_user_ticket(self, ticket):
        return None


class TvmTicketProvider(AbstractTvmTicketProvider):
    def _update_key(self):
        if not self._need_update_ticket():
            return

        try:
            self._ticket_by_destination = self._client.get_tickets(self._destinations)
        except (requests.ConnectionError, requests.Timeout) as e:
            self._logger.warn('Network problem %r', e, exc_info=True)

            if self._ticket_by_destination:
                return

            raise TvmUpdateTicketError('Some network problem {}'.format(e))

    def get_ticket(self, destination):
        # type: (Union[int, str]) -> str
        self._update_key()

        d_str = str(destination)

        if d_str not in self._ticket_by_destination and destination not in self._ticket_by_destination:
            self._logger.warn(
                'Unknown destination "%r", existing destinations: %r',
                destination,
                self._ticket_by_destination.keys(),
                exc_info=True,
            )
            raise TvmUnknownDestinationError(destination)

        return self._ticket_by_destination.get(d_str, self._ticket_by_destination.get(destination))

    def check_service_ticket(self, ticket):
        # type: (str) -> ServiceTicket
        return self._client.parse_service_ticket(ticket)

    def check_user_ticket(self, ticket):
        # type: (str) -> UserTicket
        return self._client.parse_user_ticket(ticket)


class ProviderFabric(object):
    def __init__(self):
        self._instance = None

    def create(self, settings=None, fake=False, source_id=None, secret=None, destinations=None, renew=False,
               blackbox_env=BlackboxEnv.Test, timeout=None):
        # type: (Any, bool, Optional[int], str, List[Union[int,str]], bool, BlackboxEnv, Optional[float]) -> AbstractTvmTicketProvider
        if self._instance is not None and not renew:
            return self._instance
        logger = logging.getLogger('tvm_provider')

        actual_timeout = timeout or DEFAULT_TIMEOUT
        if (settings and not settings.ENABLE_TVM) or fake:
            self._instance = FakeTvmTicketProvider(
                client=None,
                destinations=[],
                logger=logger,
            )
        else:
            if os.getenv('QLOUD_TVM_TOKEN'):
                client = QloudTvmClient(
                    source_id=settings.TVM_NAME if settings else source_id,
                    token=os.getenv('QLOUD_TVM_TOKEN'),
                    logger=logger,
                    timeout=actual_timeout,
                )
            elif os.getenv('TVMTOOL_LOCAL_AUTHTOKEN'):
                client = DeployTvmClient(
                    source_id=settings.TVM_NAME if settings else source_id,
                    token=os.getenv('TVMTOOL_LOCAL_AUTHTOKEN'),
                    logger=logger,
                    timeout=actual_timeout
                )
            elif os.getenv('TVM_TOOL_TOKEN'):
                client = TvmToolClient(
                    source_id=settings.TVM_NAME if settings else source_id,
                    url=os.getenv('TVM_TOOL_URL'),
                    token=os.getenv('TVM_TOOL_TOKEN'),
                    logger=logger,
                    timeout=actual_timeout,
                )
            else:
                if timeout is not None:
                    logger.warn("Timeout is not supported for TvmClient")
                client = TvmClient(
                    source_id=settings.TVM_NAME if settings else int(source_id),
                    secret=settings.TVM_TOKEN if settings else secret,
                    logger=logger,
                    blackbox_env=getattr(settings, 'BLACKBOX_ENV', blackbox_env),
                    destinations=[int(d) for d in destinations],
                )
            self._instance = TvmTicketProvider(
                client=client,
                destinations=settings.TVM_DESTINATIONS if settings else destinations,
                logger=logger,
            )

        return self._instance


provider_fabric = ProviderFabric()
