# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging  # noqa
from abc import ABCMeta, abstractmethod
from typing import Optional, Dict, List, Union  # noqa

import requests
from requests.packages.urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter

try:
    import ticket_parser2_py3 as ticket_parser2
except ImportError:
    import ticket_parser2

from travel.library.python.tvm_ticket_provider.models import ServiceTicket, UserTicket


class AbstractTvmClient(object):
    __metaclass__ = ABCMeta

    def __init__(self, logger):
        # type: (logging.Logger) -> None
        self._logger = logger

    @abstractmethod
    def get_tickets(self, destinations):
        # type: (List[int]) -> Dict[str][str]
        pass

    @abstractmethod
    def parse_service_ticket(self, ticket):
        # type: (str) -> ServiceTicket
        pass

    @abstractmethod
    def parse_user_ticket(self, ticket):
        # type: (str) -> UserTicket
        pass


class TvmClient(AbstractTvmClient):
    API_URL = 'https://tvm-api.yandex.net/2'

    def __init__(self, source_id, secret, logger, blackbox_env=None, destinations=None):
        # type: (int, str, logging.Logger, Optional[ticket_parser2.BlackboxEnv], Optional[List[int]]) -> None
        super(TvmClient, self).__init__(logger)
        self._tvm_client = ticket_parser2.TvmClient(ticket_parser2.TvmApiClientSettings(
            self_client_id=source_id,
            self_secret=secret,
            enable_service_ticket_checking=True,
            enable_user_ticket_checking=blackbox_env,
            dsts=destinations,
        ))

    def get_ticket(self, destination):
        # type: (Union[str, int]) -> str
        try:
            return self._tvm_client.get_service_ticket_for(client_id=int(destination))
        except ValueError:
            pass

        return self._tvm_client.get_service_ticket_for(alias=destination)

    def get_tickets(self, destinations):
        # type: (List[Union[str, int]]) -> Dict[Union[str]][str]
        return {
            dst: self.get_ticket(dst)
            for dst in destinations
        }

    def parse_service_ticket(self, ticket):
        parsed_ticket = self._tvm_client.check_service_ticket(ticket)
        return ServiceTicket(
            src=parsed_ticket.src,
            scopes=parsed_ticket.scopes,
            debug_string=parsed_ticket.debug_info,
        )

    def parse_user_ticket(self, ticket):
        parsed_ticket = self._tvm_client.check_user_ticket(ticket)
        return UserTicket(
            scopes=parsed_ticket.scopes,
            uids=parsed_ticket.uids,
            default_uid=parsed_ticket.default_uid,
            debug_string=parsed_ticket.debug_info,
        )


class TvmToolClient(AbstractTvmClient):
    def __init__(self, source_id, url, token, logger, timeout=0.01, check_user_ticket_timeout=0.1):
        # type: (int, str, str, logging.Logger, Optional[float], Optional[float]) -> None
        super(TvmToolClient, self).__init__(logger)
        self._url = url
        self._timeout = timeout
        self._check_user_ticket_timeout = check_user_ticket_timeout
        self._source_id = source_id
        self._token = token
        self._session = requests.Session()
        self._session.mount('http://', HTTPAdapter(max_retries=Retry(
            total=3,
            read=3,
            connect=3,
            backoff_factor=0.01,  # 0ms 20ms 40ms
            status_forcelist=(500, 502, 504),
        )))

    def _get(self, method, params=None, headers=None, timeout=None):
        # type: (str, Optional[dict], Optional[dict], Optional[float]) -> dict
        if headers is None:
            headers = {}
        headers['Authorization'] = self._token

        response = self._session.get(
            '{api_url}{method}'.format(api_url=self._url, method=method),
            params=params,
            headers=headers,
            timeout=timeout or self._timeout,
        )

        response.raise_for_status()

        return response.json()

    def get_tickets(self, destinations):
        # type: (List[str]) -> Dict[str][str]
        data = self._get('/tickets', {
            'src': self._source_id,
            'dsts': ','.join(str(x) for x in destinations),
        })

        tickets = {}
        for name, item in data.items():
            # add qloud aliases and tvm ids since they are interchangeable
            tickets[name] = item['ticket']
            tickets[str(item['tvm_id'])] = item['ticket']

        return tickets

    def parse_service_ticket(self, ticket):
        # type: (str) -> ServiceTicket
        data = self._get('/checksrv', headers={
            'X-Ya-Service-Ticket': ticket,
        })

        return ServiceTicket(
            src=int(data['src']),
            dst=int(data['dst']),
            scopes=data['scopes'],
            debug_string=data['debug_string'],
            logging_string=data['logging_string'],
        )

    def parse_user_ticket(self, ticket):
        # type: (str) -> UserTicket
        data = self._get(
            '/checkusr',
            headers={
                'X-Ya-User-Ticket': ticket,
            },
            timeout=self._check_user_ticket_timeout,
        )

        return UserTicket(
            default_uid=int(data['default_uid']),
            uids=[int(uid) for uid in data['uids']],
            scopes=data['scopes'],
            debug_string=data['debug_string'],
            logging_string=data['logging_string'],
        )


class QloudTvmClient(TvmToolClient):
    API_URL = 'http://localhost:1/tvm'

    def __init__(self, source_id, token, logger, timeout=0.01, check_user_ticket_timeout=0.1):
        # type: (int, str, logging.Logger, Optional[float], Optional[float]) -> None
        super(QloudTvmClient, self).__init__(
            source_id=source_id,
            url=self.API_URL,
            token=token,
            logger=logger,
            timeout=timeout,
            check_user_ticket_timeout=check_user_ticket_timeout,
        )


class DeployTvmClient(TvmToolClient):
    API_URL = 'http://localhost:2/tvm'

    def __init__(self, source_id, token, logger, timeout=0.01, check_user_ticket_timeout=0.1):
        # type: (int, str, logging.Logger, Optional[float], Optional[float]) -> None
        super(DeployTvmClient, self).__init__(
            source_id=source_id,
            url=self.API_URL,
            token=token,
            logger=logger,
            timeout=timeout,
            check_user_ticket_timeout=check_user_ticket_timeout,
        )
