# -*- coding: utf-8 -*-

from datetime import datetime, time

import six
from lxml import etree

if six.PY3:
    from html.parser import HTMLParser
else:
    from HTMLParser import HTMLParser


NBSP = u'\xa0'
_not_specified = object()


class XmlUtilsError(Exception):
    pass


class SubTagNotFound(XmlUtilsError):
    pass


class IncorrectValue(XmlUtilsError):
    pass


def get_bool_from_sub_tag(el, subtag_name, default=_not_specified, nsmap=None,
                          if_not_found=_not_specified, if_text_is_empty=_not_specified, if_invalid=_not_specified):
    if default is not _not_specified:
        if_not_found = default
        if_text_is_empty = default

    subtag = el.find('./' + subtag_name, nsmap)
    if subtag is None:
        if if_not_found is _not_specified:
            raise SubTagNotFound('Subtag {} not found in tag {}'.format(subtag_name, el.tag))
        else:
            return if_not_found

    if not subtag.text:
        if if_text_is_empty is not _not_specified:
            return if_text_is_empty

    return _get_bool_from_text(subtag.text, if_invalid=if_invalid)


def _get_bool_from_text(text, if_invalid=_not_specified):
    text = (text or '').strip()

    if text in ('1', 'true'):
        return True

    if text in ('0', 'false'):
        return False

    if if_invalid is not _not_specified:
        return if_invalid

    raise IncorrectValue('XML Boolean valid values are "0", "1", "false", "true"')


def get_sub_tag_text(el, subtag_name, default=_not_specified, nsmap=None,
                     if_not_found=_not_specified, if_text_is_empty=_not_specified):
    """
    Возвращает содержимое дочернего элемента
        <a>
          <b>asdf</b>
        </a>
        get_sub_tag_text(a_el, 'b') == 'asdf'
    """
    if default is not _not_specified:
        if_not_found = default
        if_text_is_empty = default

    subtag = el.find('./' + subtag_name, nsmap)
    if subtag is None:
        if if_not_found is _not_specified:
            raise SubTagNotFound('Subtag {} not found in tag {}'.format(subtag_name, el.tag))
        else:
            return if_not_found

    if if_text_is_empty is _not_specified:
        return subtag.text

    return if_text_is_empty if subtag.text is None else subtag.text


def get_sub_tag_text2(tag, tagname, default=_not_specified, nsmap=None):
    """
    Упрощенная версия get_sub_tag_text
    Тег может быть пустым, без текста, в этом случае text == None, но функция
    возвращает u'' или default, если указано.
    """

    if_empty_text = u'' if default is _not_specified else default

    subtag = tag.find('./' + tagname, nsmap)
    try:
        return subtag.text or if_empty_text
    except AttributeError:
        if default is _not_specified:
            raise
        else:
            return default


def copy_lxml_element(el, top_element_only=False):
    el = etree.fromstring(etree.tostring(el))

    if top_element_only:
        del el[:]

    return el


def lxml_get_text(element):
    return u''.join(element.xpath('.//text()')).strip()


def lxml_get_date(element, format='%Y-%m-%d'):
    try:
        return datetime.strptime(lxml_get_text(element), format).date()
    except ValueError:
        pass


def lxml_get_time(element, format=None):
    text = lxml_get_text(element)
    try:
        if format is None:
            return time(*[int(t.strip()) for t in text.split(':')])
        else:
            return datetime.strptime(lxml_get_text(element), format).time()
    except ValueError:
        pass


def lxml_to_unicode(element):
    return etree.tounicode(element, pretty_print=True)


def lxml_humanize(element, **kwargs):
    data = etree.tounicode(element, pretty_print=True, **kwargs)
    return HTMLParser().unescape(data)
