# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from argparse import ArgumentParser
from functools import wraps
import logging

from library.python.vault_client import instances
from six import string_types


LOG = logging.getLogger(__name__)


def with_retries(func=None, max_attempts=3, exc_info=True, counter=None, key=None):
    if not func:
        return lambda f: with_retries(f, max_attempts=max_attempts, exc_info=exc_info, counter=counter, key=key)

    @wraps(func)
    def wrapper(*args, **kwargs):
        attempts = max_attempts
        while True:
            try:
                return func(*args, **kwargs)
            except Exception as e:
                if counter is not None and key is not None:
                    counter[key] += 1
                attempts -= 1
                if attempts == 0:
                    raise
                LOG.warn('%r failed, %d attempts left', func, attempts)
                if exc_info:
                    LOG.warn(e)

    return wrapper


def get_options(args):
    parser = ArgumentParser()
    parser.add_argument('--vault-token')
    return parser.parse_known_args(args)


@with_retries()
def get_arg_value(vault_client, secret_key, value_key):
    return str(vault_client.get_version(secret_key)['value'][value_key])


def get_filled_args(args, vault_client):
    filled_args = list()
    for arg in args:
        filled_args.append(get_filled_arg(arg, vault_client))
    return filled_args


def get_filled_arg(arg, vault_client):
    if not arg.startswith('sec-'):
        return arg
    parts = arg.split('.')
    if len(parts) > 2:
        raise ValueError('Nested keys not allowed: {}'.format(arg))
    if len(parts) == 1:
        parts.append('value')
    secret_key, value_key = parts
    try:
        arg = get_arg_value(vault_client, secret_key, value_key)
    except Exception as e:
        raise Exception('Failed to resolve secret "{}": {}'.format(arg, e))
    return arg


def resolve_secrets(args):
    options, rest_args = get_options(args)
    if options.vault_token is None:
        raise ValueError('Token option "--vault-token" not found')
    client = instances.Production(authorization=options.vault_token)
    return get_filled_args(rest_args, client)


def resolve_secrets_ns(options):
    client = instances.Production(authorization=options.vault_token)
    for key in options.__dict__:
        value = options.__dict__[key]
        if isinstance(value, string_types):
            options.__dict__[key] = get_filled_arg(value, client)
