# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import gzip
import os
import subprocess
import tempfile
import time
import traceback
from contextlib import closing
from StringIO import StringIO

import library.python.resource
import MySQLdb
from travel.library.recipe_utils.utils import log, Timer


def load_schema(dump_archive_path, db, user, passwd, host, port, mysql_executable=None):
    dump_file = tempfile.mktemp()
    with Timer('Unarchiving dump {} to {}'.format(dump_archive_path, dump_file)):
        dump_arch = library.python.resource.find(dump_archive_path)
        if dump_arch is None:
            raise Exception("Can't find resource: {}".format(dump_archive_path))

        buf = StringIO(dump_arch)
        with gzip.GzipFile(fileobj=buf) as zfile:
            dump_content = zfile.read()
            with open(dump_file, 'w') as f:
                f.write(dump_content)

    with Timer('Creating db {} on {}:{}'.format(db, host, port)):
        try_count = 10  # Mysql требуется время, чтобы стартовать
        for i in range(1, try_count + 1):
            try:
                with closing(MySQLdb.connect(host=host, user=user, passwd=passwd, connect_timeout=2, port=port)) as conn:
                    cursor = conn.cursor()
                    cursor.execute('DROP DATABASE IF EXISTS {}'.format(db))
                    cursor.execute('CREATE DATABASE {} DEFAULT CHARSET utf8;'.format(db))
                break
            except Exception as ex:
                log('Create db. Try %s of %s. Error!\n%s' % (str(i), str(try_count), repr(ex)))
                if i == try_count:
                    raise
            time.sleep(5)

    with Timer('Loading dump from {}'.format(dump_file)):
        # TODO:lorekhov тут имеет смысл предусмотреть механизм позволяющий явно указывать откуда мы берем mysql
        mysql_dir = os.getenv('RASP_MYSQL_DIR', os.path.join(os.getcwd(), 'mysql'))
        binary_mysql = os.path.join(mysql_dir, 'bin', 'mysql')
        for executable in ['mysql', mysql_executable, binary_mysql]:
            if not executable:
                continue

            log('Trying to load dump with:', executable)
            try:
                subprocess.check_call(
                    'cat {schema_file} | {executable} -u {user} --password={passwd} -h {host} --port={port} {db}'.format(
                        db=db,
                        executable=executable,
                        schema_file=dump_file,
                        host=host,
                        port=port,
                        user=user,
                        passwd=passwd,
                    ),
                    shell=True,
                    stderr=subprocess.STDOUT,
                )
            except Exception as ex:
                log(repr(ex.output), traceback.format_exc())
            else:
                break

    return dump_file
