# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
import socket
import subprocess
import sys
import tarfile
import time
import datetime

import yatest
from library.python.testing.recipe import set_env


class Timer(object):
    def __init__(self, description):
        self.description = description

    def __enter__(self):
        log("{description}".format(description=self.description))
        self.start = time.time()

    def __exit__(self, type, value, traceback):
        self.end = time.time()
        value = self.end - self.start
        log("DONE: {description} {value}".format(description=self.description, value=value))


def untar(file_path, path='.'):
    log('Un-tar file {}'.format(file_path))
    tar = tarfile.open(file_path)
    tar.extractall(path)
    tar.close()


def run_process(command, process_name, env):
    command = map(str, command)
    log('Running command:', ' '.join(command))
    process = subprocess.Popen(
        command,
        stdout=open(yatest.common.output_path('{}.out.log'.format(process_name)), 'w'),
        stderr=open(yatest.common.output_path('{}.err.log'.format(process_name)), 'w'),
        env=env
    )
    with open('{}.pid'.format(process_name), 'w') as f:
        f.write(str(process.pid))

    return process


def wait_for_process(process, process_name, is_alive, attempts=60):
    for attempt in range(attempts):
        log(process_name, 'attempt', attempt, '...')
        result = process.poll()
        if result is not None:
            log('process {process_name} failed. stderr is {stderr}, stdout is {stdout}, retcode is {retcode}, pid is {pid}'.format(
                process_name=process_name,
                stderr=process.stderr,
                stdout=process.stdout,
                retcode=result,
                pid=process.pid
            ))
            raise Exception('Could not launch {}'.format(process_name))

        if is_alive():
            return
        time.sleep(1)
    raise Exception('Failed to start {}'.format(process_name))


def is_alive(port, process_name):
    addr = socket.getaddrinfo('localhost', str(port), socket.AF_INET, socket.SOCK_STREAM)
    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    sock.settimeout(1)
    try:
        sock.connect(addr[0][4])
    except socket.error:
        return False
    else:
        log('{} is up on port {})'.format(process_name, port))
        return True


def is_port_available(port):
    try:
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.bind(('', port))
        sock.close()
        return True
    except Exception as ex:
        log('Error while binding port {}: {}'.format(port, ex))
    return False


def get_free_port(desired_port=None):
    if desired_port and is_port_available(desired_port):
        return desired_port

    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    sock.bind(('', 0))
    port = sock.getsockname()[1]
    sock.close()
    return port


def set_environ(key, value):
    set_env(key, value)
    os.environ[key] = value


def copy_environ(src, dst):
    value = os.getenv(src)
    set_environ(dst, value)


def log(*args):
    print(datetime.datetime.now(), *args)
    sys.stdout.flush()


def check_call(cmd, process_name, env):
    log('Command is:', ' '.join(cmd))
    subprocess.check_call(
        cmd,
        stdout=open(yatest.common.output_path('{}.out.log'.format(process_name)), 'w'),
        stderr=open(yatest.common.output_path('{}.err.log'.format(process_name)), 'w'),
        env=env
    )
