# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import getpass
import hashlib
import os
import random
import signal
import subprocess
import sys

from devtools.ya.test.programs.test_tool.lib import tmpfs

import yatest
from travel.library.recipe_utils.utils import (
    is_alive, log, run_process, Timer, untar, wait_for_process, get_free_port, set_environ
)


description = 'mysql-step'

_MYSQL_DIST_DIR = 'mysql-5.7.29'


def run_mysql(uniq):
    random.seed(uniq)
    mysql_port = get_free_port(desired_port=3306 + random.randrange(1000))
    log('Starting mysql on port:', mysql_port)

    mysql_dir = os.path.join(os.getcwd(), uniq, _MYSQL_DIST_DIR)
    mysql_data_dir = os.path.join(mysql_dir, 'data-dir')
    mysql_bin = os.path.join(mysql_dir, 'bin')
    mysqld_path = os.path.join(mysql_dir, 'bin', 'mysqld')
    os.mkdir(mysql_data_dir)
    try:
        tmpfs.mount_tempfs_newns(mysql_data_dir, 2000)
    except:
        pass

    pid_file = 'mysql_{}.pid'.format(uniq)
    sock_file = './mysql-{}-sock'.format(uniq)
    set_environ('TRAVEL_MYSQL_RECIPE_DATA_DIR', mysql_data_dir)
    set_environ('TRAVEL_MYSQL_RECIPE_HOST', '127.0.0.1')
    set_environ('TRAVEL_MYSQL_RECIPE_PORT', str(mysql_port))
    set_environ('TRAVEL_MYSQL_RECIPE_TESTS_DB', 'travel_tests_{}'.format(uniq))
    set_environ('TRAVEL_MYSQL_RECIPE_PASSWORD', '')
    set_environ('TRAVEL_MYSQL_PID', pid_file)
    set_environ('TRAVEL_MYSQL_DIR', mysql_dir)
    set_environ('TRAVEL_MYSQL_BIN', mysql_bin)

    env = os.environ.copy()
    env.update({'LD_LIBRARY_PATH': mysql_bin})

    process_name = 'mysql_{}'.format(uniq)
    command = [
        mysqld_path,
        '--no-defaults',
        '--port', str(mysql_port),
        '--socket', sock_file,
        '--datadir', mysql_data_dir,
        '--pid-file', pid_file,
        '--sql-mode', 'NO_ENGINE_SUBSTITUTION',
        '--language={}/share/english/'.format(mysql_dir),
        '--lc-messages-dir={}/share/'.format(mysql_dir),
        '--skip-performance-schema',
        '--show-compatibility-56=ON',
        '--log-error-verbosity=2',
        '--innodb-use-native-aio=0',
    ]

    with Timer('Initializing mysql...:'):
        install_command = command + [
            '--initialize-insecure',
            '--basedir={}'.format(mysql_dir),
            '--user={}'.format(getpass.getuser()),
        ]
        log('Command is:', ' '.join(install_command))
        subprocess.check_call(
            install_command,
            stdout=sys.stdout, stderr=subprocess.STDOUT, env=env
        )

    with Timer('Starting mysql...'):
        process_name = 'mysql_{}'.format(uniq)
        process = run_process(command + ['--skip-grant-tables'], process_name, env)

        wait_for_process(process, process_name, lambda: is_alive(mysql_port, process_name))

    log('Mysql successfully started')


def setup():
    uniq = hashlib.md5(os.getcwd().encode("utf-8")).hexdigest()

    mysql_dir = os.path.join(os.getcwd(), uniq)
    try:
        os.makedirs(mysql_dir)
    except OSError:
        pass

    mysql_archive_path = yatest.common.build_path('travel/library/recipes/packages/mysql/mysql-5.7.29.tar.gz')
    untar(mysql_archive_path, path=mysql_dir)

    run_mysql(uniq)


def teardown():
    pid_file = os.getenv('TRAVEL_MYSQL_PID', 'mysql.pid')
    with open(pid_file, 'r') as f:
        pid = int(f.read())
        os.kill(pid, signal.SIGTERM)
    mysql_data_dir = os.getenv('TRAVEL_MYSQL_RECIPE_DATA_DIR')
    try:
        tmpfs.unmount(mysql_data_dir)
    except:
        pass
