import argparse
import logging

from abc import ABC, abstractmethod
from time import time

from django.db import transaction

from travel.library.python.dicts.country_repository import CountryRepository
from travel.library.python.dicts.region_repository import RegionRepository
from travel.library.python.dicts.settlement_repository import SettlementRepository
from travel.marketing.content.admin.www.models import Country, Settlement, Region

log = logging.getLogger(__name__)


class BaseLoader(ABC):
    @property
    @abstractmethod
    def model(self):
        pass

    @property
    def model_name(self):
        return self.model.__name__

    @property
    @abstractmethod
    def repository(self):
        pass

    @property
    @abstractmethod
    def update_fields(self):
        pass

    @abstractmethod
    def build(self, value):
        pass

    @abstractmethod
    def update(self, old, new):
        pass

    def get_values_from_file(self, path):
        repository = self.repository()
        repository.load_from_file(path)

        return repository.itervalues()

    def load(self, path):
        log.info("Start loading {}".format(self.model_name))
        start = time()

        old_objects = {s.id: s for s in self.model.objects.all()}
        add_objects = []
        update_objects = []

        for item in self.get_values_from_file(path):
            new_obj = self.build(item)
            if not new_obj:
                continue

            old_obj = old_objects.pop(new_obj.id, None)
            if not old_obj:
                add_objects.append(new_obj)
            else:
                obj = self.update(old_obj, new_obj)
                if obj:
                    update_objects.append(obj)

        hide_objects = list(old_objects.values())
        for obj in hide_objects:
            obj.hidden = True

        log.info("{} add: {}, update: {}, hide: {}".format(self.model_name, len(add_objects), len(update_objects),
                                                             len(hide_objects)))

        self.model.objects.bulk_create(add_objects)
        self.model.objects.bulk_update(update_objects, self.update_fields)
        self.model.objects.bulk_update(hide_objects, ["hidden"])

        run_time = time() - start
        log.info("Loaded {} in {:.4}s".format(self.model_name, run_time))


class SettlementLoader(BaseLoader):
    MAX_MAJORITY_INDEX = 4  # не грузим города с важностью "Посёлок или деревня"

    @property
    def model(self):
        return Settlement

    @property
    def repository(self):
        return SettlementRepository

    @property
    def update_fields(self):
        return ["title", "slug", "country", "region", "hidden"]

    def build(self, value):
        if value.Majority > self.MAX_MAJORITY_INDEX:
            return None

        settlement = Settlement(
            id=value.Id,
            title=value.Title.Ru.Nominative,
            slug=value.Slug
        )

        if value.CountryId:
            settlement.country_id = value.CountryId

        if value.RegionId:
            settlement.region_id = value.RegionId

        return settlement

    def update(self, old, new):
        if old.title != new.title or old.slug != new.slug or old.country != new.country or old.region != new.region:
            old.title = new.title
            old.slug = new.slug
            old.country = new.country
            old.region = new.region
            old.hidden = False

            return old


class CountryLoader(BaseLoader):
    @property
    def model(self):
        return Country

    @property
    def repository(self):
        return CountryRepository

    @property
    def update_fields(self):
        return ["title", "hidden"]

    def build(self, value):
        return Country(
            id=value.Id,
            title=value.Title.Ru.Nominative
        )

    def update(self, old, new):
        if old.title != new.title:
            old.title = new.title
            old.hidden = False

            return old


class RegionLoader(BaseLoader):
    @property
    def model(self):
        return Region

    @property
    def repository(self):
        return RegionRepository

    @property
    def update_fields(self):
        return ["title", "hidden"]

    def build(self, value):
        return Region(
            id=value.Id,
            title=value.TitleNominative.Ru
        )

    def update(self, old, new):
        if old.title != new.title:
            old.title = new.title
            old.hidden = False

            return old


def update(countries_path, settlements_path, regions_path):
    with transaction.atomic():
        CountryLoader().load(countries_path)
        RegionLoader().load(regions_path)
        SettlementLoader().load(settlements_path)

    log.info("All models loaded")


def run_cli():
    parser = argparse.ArgumentParser()
    parser.add_argument("--settlements", required=True)
    parser.add_argument("--countries", required=True)
    parser.add_argument("--regions", required=True)

    args = parser.parse_args()

    update(args.countries, args.settlements, args.regions)


if __name__ == "__main__":
    run_cli()
