from django.db import models
from slugify import slugify

from travel.marketing.content.admin.admin.mds import mds_s3_media_storage
from travel.avia.library.python.avatars.storage import AvatarsPngStorage


class Country(models.Model):
    id = models.IntegerField(primary_key=True)
    title = models.CharField(verbose_name="наименование", null=False, blank=False, max_length=100, db_index=True)
    hidden = models.BooleanField(verbose_name="скрывать", null=False, blank=False, default=False)

    def __str__(self):
        return self.title

    class Meta:
        verbose_name = "Страна"
        verbose_name_plural = "Страны"
        app_label = "www"
        db_table = "www_dict_country"


class Region(models.Model):
    id = models.IntegerField(primary_key=True)
    title = models.CharField(verbose_name="наименование", null=False, blank=False, max_length=100, db_index=True)
    hidden = models.BooleanField(verbose_name="скрывать", null=False, blank=False, default=False)

    def __str__(self):
        return self.title

    class Meta:
        verbose_name = "Регион"
        verbose_name_plural = "Регионы"
        app_label = "www"
        db_table = "www_dict_region"


class Settlement(models.Model):
    id = models.IntegerField(primary_key=True)
    title = models.CharField(verbose_name="наименование", null=False, blank=False, max_length=100)
    slug = models.SlugField(max_length=100, verbose_name="Название для ссылок", db_index=True)  # для поиска в админке
    country = models.ForeignKey(Country, on_delete=models.DO_NOTHING, verbose_name="страна", null=True, blank=True)
    region = models.ForeignKey(Region, on_delete=models.DO_NOTHING, verbose_name="регион", null=True, blank=True)
    hidden = models.BooleanField(verbose_name="скрывать", null=False, blank=False, default=False)

    def __str__(self):
        parts = []
        if self.region:
            parts.append(str(self.region))

        if self.country:
            parts.append(str(self.country))

        return str(self.title) + " ({})".format(", ".join(parts))

    class Meta:
        verbose_name = "Город"
        verbose_name_plural = "Города"
        app_label = "www"
        db_table = "www_dict_settlement"


class BaseContent(models.Model):
    name = models.CharField(verbose_name="наименование", null=False, blank=False, max_length=255, db_index=True)

    title = models.CharField(verbose_name="title", null=False, blank=False, max_length=255)
    seo_description = models.TextField(verbose_name="description", null=False, blank=False)
    slug = models.SlugField(verbose_name="slug", help_text='Название для ссылок',
                            null=False, blank=True, max_length=100, unique=True)

    og_title = models.CharField(verbose_name="og:title", null=False, blank=True, max_length=255)
    og_description = models.TextField(verbose_name="og:description", null=False, blank=True)
    og_image = models.ImageField(verbose_name="og:image", blank=True,
                                 storage=mds_s3_media_storage, upload_to="images/og")

    twitter_title = models.CharField(verbose_name="twitter:title", null=False, blank=True, max_length=255)
    twitter_description = models.TextField(verbose_name="twitter:description", null=False, blank=True)
    twitter_image = models.ImageField(verbose_name="twitter:image", blank=True,
                                      storage=mds_s3_media_storage, upload_to="images/twitter")

    def save(self, *args, **kwargs):
        if not self.slug:
            slug = slugify(self.name)
            counter = 1
            while type(self).objects.filter(slug=slug).count():
                slug = slugify(self.name) + "-{}".format(counter)

            self.slug = slug

        super(BaseContent, self).save(*args, **kwargs)

    class Meta:
        abstract = True


class City(BaseContent):
    settlement = models.ForeignKey(
        Settlement,
        on_delete=models.DO_NOTHING,
        null=False, blank=False,
        verbose_name="город"
    )

    info_box = models.TextField(verbose_name="информационный блок", null=False, blank=False)
    image = models.ImageField(verbose_name="изображение", blank=False,
                              storage=mds_s3_media_storage, upload_to="img/city")
    avatars_image = models.ImageField(
        verbose_name="Ссылка на изображение в Аватарнице",
        blank=False,
        default='',
        upload_to='',
        storage=AvatarsPngStorage(),
    )

    description = models.TextField(verbose_name="описание", null=False, blank=False)

    def __str__(self):
        return self.name

    class Meta:
        verbose_name = "Страница города"
        verbose_name_plural = "Страницы города"
        app_label = "www"
        db_table = "www_city"


class Attraction(BaseContent):
    city = models.ForeignKey(
        City,
        on_delete=models.DO_NOTHING,
        null=False, blank=False,
        verbose_name="город"
    )

    image = models.ImageField(verbose_name="изображение", blank=False,
                              storage=mds_s3_media_storage, upload_to="img/attraction")

    description = models.TextField(verbose_name="описание", null=False, blank=False)
    majority = models.IntegerField(verbose_name="приоритет внутри города", null=False, blank=False)
    preview_text = models.CharField(verbose_name="текст для превью карусели", null=False, blank=False, max_length=120)

    def __str__(self):
        return self.name

    class Meta:
        verbose_name = "Страица достопримечательности"
        verbose_name_plural = "Страницы достопримечательностей"
        app_label = "www"
        db_table = "www_attractions"


class Landing(BaseContent):
    image = models.ImageField(verbose_name="изображение", blank=False,
                              storage=mds_s3_media_storage, upload_to="img/content")

    description = models.TextField(verbose_name="описание", null=False, blank=False)

    def __str__(self):
        return self.name

    class Meta:
        verbose_name = "Страница контентного лендинга"
        verbose_name_plural = "Страницы контентного лендинга"
        app_label = "www"
        db_table = "www_landing"


class SeoPage(BaseContent):
    image = models.ImageField(verbose_name="изображение", blank=False,
                              storage=mds_s3_media_storage, upload_to="img/content")

    parent = models.ForeignKey(
        "self",
        on_delete=models.DO_NOTHING,
        null=True, blank=True,
        verbose_name="Родительская страница"
    )

    description = models.TextField(verbose_name="описание", null=False, blank=False)
    use_navigation = models.BooleanField(verbose_name="использовать h2 как навигацию", null=False, blank=False)

    def __str__(self):
        return self.name

    class Meta:
        verbose_name = "Страница SEO справочника"
        verbose_name_plural = "Страницы SEO справочника"
        app_label = "www"
        db_table = "www_seo"


class Image(models.Model):
    source = models.ImageField(verbose_name="изображение", blank=False,
                               height_field="height",
                               width_field="width",
                               storage=mds_s3_media_storage, upload_to="img/common")

    width = models.IntegerField("ширина", editable=False, null=False)
    height = models.IntegerField("высота", editable=False, null=False)

    def __str__(self):
        return "Изображение " + str(self.source)

    class Meta:
        verbose_name = "Изображение"
        verbose_name_plural = "Изображения"
        app_label = "www"
        db_table = "www_images"
