package main

import (
	"context"
	"fmt"
	"sync"

	"github.com/jonboulle/clockwork"
	gcodes "google.golang.org/grpc/codes"
	ghealth "google.golang.org/grpc/health"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/configuration/backends/yav"
	"a.yandex-team.ru/travel/library/go/grpcgateway"
	grpcmetrics "a.yandex-team.ru/travel/library/go/grpcutil/metrics"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	"a.yandex-team.ru/travel/library/go/grpcutil/service/servingstatus"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
	"a.yandex-team.ru/travel/library/go/vault"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/app"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/geo"
	geoHandler "a.yandex-team.ru/travel/marketing/content/backend/internal/geo/handler"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/landing"
	landingHandler "a.yandex-team.ru/travel/marketing/content/backend/internal/landing/handler"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/db"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/seo"
	seoHandler "a.yandex-team.ru/travel/marketing/content/backend/internal/seo/handler"
	contentv1 "a.yandex-team.ru/travel/marketing/content/v1"
)

const (
	serviceName = "content-backend"
)

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())

	secretsResolver := vault.NewYavSecretsResolverFromEnv("RASP_VAULT_OAUTH_TOKEN")
	config := configuration.NewConfitaLoader(
		configuration.WithYamlDefaultBackend(),
		configuration.WithYavBackend(secretsResolver, yav.SecUIDConfig{
			Production: "sec-01es5vrx9n37eq71cmngs8xkr9",
			Testing:    "sec-01es0mr49rk73zfmvhdren9dvv",
		}),
	)

	err := config.Load(ctx, &app.Cfg)
	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.NewDeploy(&app.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)
	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
		ctxCancel()
	}()

	var tvmClient tvm.Client
	tvmAllowedIds := tvmutil.TvmClientIDFromInt(app.Cfg.Tvm.WhiteList)
	if app.Cfg.Tvm.Enabled {
		tvmClient, err = tvmtool.NewDeployClient()
		if err != nil {
			fmt.Println("failed to create tvm client:", err)
		}
	}
	rootRegistry := metrics.NewRegistryWithDeployTags()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, app.Cfg.Metrics.PerfMetricsRefreshInterval)

	pgpool, err := db.New(&app.Cfg.PGAAS, logger)
	if err != nil {
		fmt.Println("can not connect to postgres cluster, err:", err)
		return
	}

	blogService := landing.NewService(logger, pgpool, app.Cfg.S3)
	grpcLandingHandler := landingHandler.NewGRPCBlogContentHandler(logger, blogService)

	geoService := geo.NewService(logger, pgpool, app.Cfg.S3)
	grpcGeoHandler := geoHandler.NewGRPCGeoContentHandler(logger, geoService)

	seoService := seo.NewService(logger, pgpool, app.Cfg.S3)
	grpcSeoHandler := seoHandler.NewGRPCSeoContentHandler(logger, seoService)

	grpcServiceRegisterers := []grpcserver.ServiceRegisterer{
		grpcLandingHandler.GetServiceRegisterer(),
		grpcGeoHandler.GetServiceRegisterer(),
		grpcSeoHandler.GetServiceRegisterer(),
	}

	healthServer := ghealth.NewServer()
	servingStatusService := servingstatus.NewService(
		logger,
		serviceName,
		healthServer.SetServingStatus,
		app.Cfg.HealthCheck.UpdateInterval,
		clockwork.NewRealClock(),
	).Requires(
		func() bool {
			_, err := pgpool.GetReadableConnection()
			return err == nil
		},
	)
	servingStatusService.MonitorServingStatus(ctx)

	grpcGateway := grpcgateway.NewGateway(
		&app.Cfg.GrpcGateway,
		grpcgateway.NewService(serviceName, "/geo", app.Cfg.Grpc.Addr, contentv1.RegisterSeoContentServiceHandlerFromEndpoint, nil).
			WithTvm(tvmClient, app.Cfg.Tvm.SelfAppID, app.Cfg.Tvm.Enabled),
		grpcgateway.NewService(serviceName, "/seo", app.Cfg.Grpc.Addr, contentv1.RegisterGeoContentServiceHandlerFromEndpoint, nil).
			WithTvm(tvmClient, app.Cfg.Tvm.SelfAppID, app.Cfg.Tvm.Enabled),
		grpcgateway.NewService(serviceName, "/landing", app.Cfg.Grpc.Addr, contentv1.RegisterLandingContentServiceHandlerFromEndpoint, nil).
			WithTvm(tvmClient, app.Cfg.Tvm.SelfAppID, app.Cfg.Tvm.Enabled),
	)

	// running server
	wg := sync.WaitGroup{}
	wg.Add(3)

	go func() {
		defer wg.Done()
		builder := grpcserver.NewGrpcServerBuilder(app.Cfg.Grpc, logger).
			WithInterceptors(grpcmetrics.NewGrpcMetricsInterceptor(
				rootRegistry.WithPrefix("grpc"),
				grpcmetrics.WithCodesCount(gcodes.NotFound, gcodes.Internal, gcodes.Unknown))).
			WithRegisterers(grpcServiceRegisterers...)
		if tvmClient != nil {
			builder = builder.WithTVM(tvmClient, tvmAllowedIds)
		}
		server := builder.WithHealthServer(healthServer).Build()

		err = server.Run(context.Background())
		if err != nil {
			logger.Fatal("Error while starting Grpc server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = metricserver.RunMetricsHTTPServer(context.Background(), app.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = grpcGateway.Run(ctx)
		if err != nil {
			logger.Info("GRPC-gateway server closed", log.Error(err))
		}
	}()

	logger.Info("Started")
	wg.Wait()
}
