package main

import (
	"context"
	"encoding/json"
	"flag"
	"log"
	"time"

	"google.golang.org/grpc"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"
	"google.golang.org/grpc/metadata"

	contentv1 "a.yandex-team.ru/travel/marketing/content/v1"
)

func PrettyPrintResult(v interface{}) {
	content, err := json.MarshalIndent(v, "", "  ")
	if err != nil {
		log.Fatalf("Could not convert value to json %s", err)
	}

	log.Printf("Result is: %s", content)
}

func GetCity(ctx context.Context, conn *grpc.ClientConn, slug string) {
	client := contentv1.NewGeoContentServiceClient(conn)
	request := contentv1.GetContentBySlugRequest{Slug: slug}
	result, err := client.GetCityContent(ctx, &request)
	if err != nil {
		log.Fatalf("Error happens: %+v", err)
	}

	PrettyPrintResult(result)
}

func GetCityImage(ctx context.Context, conn *grpc.ClientConn, id int32) {
	client := contentv1.NewGeoContentServiceClient(conn)
	request := contentv1.GetContentByIdRequest{Id: id}
	result, err := client.GetCityImage(ctx, &request)
	if err != nil {
		log.Fatalf("Error happens: %+v", err)
	}

	PrettyPrintResult(result)
}

func GetAttraction(ctx context.Context, conn *grpc.ClientConn, slug string) {
	client := contentv1.NewGeoContentServiceClient(conn)
	request := contentv1.GetContentBySlugRequest{Slug: slug}
	result, err := client.GetAttractionContent(ctx, &request)
	if err != nil {
		log.Fatalf("Error happens: %+v", err)
	}

	PrettyPrintResult(result)
}

func GetSeoContent(ctx context.Context, conn *grpc.ClientConn, slug string) {
	client := contentv1.NewSeoContentServiceClient(conn)
	request := contentv1.GetContentBySlugRequest{Slug: slug}
	result, err := client.GetSeoContent(ctx, &request)
	if err != nil {
		log.Fatalf("Error happens: %+v", err)
	}

	PrettyPrintResult(result)
}

func GetLanding(ctx context.Context, conn *grpc.ClientConn, slug string) {
	client := contentv1.NewLandingContentServiceClient(conn)
	request := contentv1.GetContentBySlugRequest{Slug: slug}
	result, err := client.GetLanding(ctx, &request)
	if err != nil {
		log.Fatalf("Error happens: %+v", err)
	}

	PrettyPrintResult(result)
}

func CheckHealth(ctx context.Context, conn *grpc.ClientConn) {
	client := healthpb.NewHealthClient(conn)
	request := healthpb.HealthCheckRequest{Service: "content-backend"}
	result, err := client.Check(ctx, &request)
	if err != nil {
		log.Fatalf("Error happens: %+v", err)
	}

	PrettyPrintResult(result)
}

func callGrpc(serverAddr, content, slug, ticket string, id int32) {
	var opts []grpc.DialOption
	opts = append(opts, grpc.WithInsecure())

	conn, err := grpc.Dial(serverAddr, opts...)
	if err != nil {
		log.Fatalf("fail to dial: %v", err)
	}
	defer conn.Close()

	ctx := context.Background()
	if len(ticket) > 0 {
		ctx = metadata.AppendToOutgoingContext(ctx, "X-Ya-Service-Ticket", ticket)
	}

	switch content {
	case "city":
		GetCity(ctx, conn, slug)
	case "city-image":
		GetCityImage(ctx, conn, id)
	case "attraction":
		GetAttraction(ctx, conn, slug)
	case "seo":
		GetSeoContent(ctx, conn, slug)
	case "landing":
		GetLanding(ctx, conn, slug)
	case "health":
		CheckHealth(ctx, conn)
	}
}

func main() {
	content := flag.String(
		"content",
		"none",
		"type of content, possible values: {city, city-image, attraction, seo, landing}")
	grpcServerAddrPtr := flag.String(
		"grpc_server_addr",
		"localhost:9001",
		"The server address in the format of host:port")
	ticket := flag.String(
		"tvm_ticket",
		"",
		"TVM auth ticket")
	slug := flag.String(
		"slug",
		"none",
		"content slug")
	id := flag.Int64(
		"id",
		0,
		"content id")

	flag.Parse()

	start := time.Now()

	callGrpc(*grpcServerAddrPtr, *content, *slug, *ticket, int32(*id))

	elapsed := time.Since(start)
	log.Printf("Elapsed: %s", elapsed)
}
