package app

import (
	"time"

	"a.yandex-team.ru/travel/library/go/grpcgateway"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
)

type TvmConfig struct {
	Enabled   bool     `config:"tvm-enabled"`
	WhiteList []uint32 `config:"tvm-white-list"`
	SelfAppID uint32   `config:"tvm-self-app-id"`
}

type Config struct {
	EnvType     string `config:"yenv-type,required"`
	HTTP        httpserver.HTTPConfig
	Grpc        grpcserver.GrpcConfig
	GrpcGateway grpcgateway.Config
	Metrics     metricsserver.MetricsConfig
	Logging     logging.Config
	HealthCheck HealthCheckConfig
	Tvm         TvmConfig
	PGAAS       DBConfig
	S3          S3Config
}

type DBConfig struct {
	Hosts        string `config:"pgaas-hosts,required"`
	Password     string `config:"pgaas-password,required"`
	Port         uint16 `config:"pgaas-port"`
	User         string `config:"pgaas-user"`
	DataBase     string `config:"pgaas-database"`
	LogLevel     string `config:"pgaas-loglevel"`
	MaxOpenConns int    `config:"pgaas-maxconns"`
}

type HealthCheckConfig struct {
	UpdateInterval time.Duration `config:"health-check-update-interval"`
}

type S3Config struct {
	Endpoint string `config:"s3-endpoint"`
	Bucket   string `config:"s3-bucket"`
	Public   bool   `config:"s3-public"`
}

var DefaultDB = DBConfig{
	Port:         6432,
	User:         "content_admin",
	DataBase:     "content",
	LogLevel:     "error",
	MaxOpenConns: 20,
}

var DefaultHealthCheckConfig = HealthCheckConfig{
	UpdateInterval: 1 * time.Second,
}

var DefaultS3 = S3Config{
	Endpoint: "s3.mds.yandex.net",
	Bucket:   "travel-marketing-content-test-bucket",
	Public:   false,
}

var DefaultTvmConfig = TvmConfig{
	Enabled:   false,
	WhiteList: nil,
}

var Cfg = Config{
	EnvType:     "development",
	HTTP:        httpserver.DefaultHTTPConfig,
	Grpc:        grpcserver.DefaultGrpcConfig,
	GrpcGateway: grpcgateway.DefaultConfig,
	Metrics:     metricsserver.DefaultMetricsConfig,
	Logging:     logging.DefaultConfig,
	HealthCheck: DefaultHealthCheckConfig,
	Tvm:         DefaultTvmConfig,
	PGAAS:       DefaultDB,
	S3:          DefaultS3,
}
