package handler

import (
	"context"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/geo"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/repository"
	contentv1 "a.yandex-team.ru/travel/marketing/content/v1"
)

var errInternalError = status.Error(codes.Internal, "geo handler internal error")

type GRPCGeoContentHandler struct {
	service *geo.Service
	logger  log.Logger
}

func NewGRPCGeoContentHandler(l log.Logger, srv *geo.Service) *GRPCGeoContentHandler {
	return &GRPCGeoContentHandler{service: srv, logger: l}
}

func (h *GRPCGeoContentHandler) logError(err error) {
	h.logger.Error("geo handler error", log.Error(err))
}

func (h *GRPCGeoContentHandler) GetCityContent(ctx context.Context, request *contentv1.GetContentBySlugRequest) (
	response *contentv1.CityContent,
	err error,
) {
	span, ctx := opentracing.StartSpanFromContext(ctx, "internal.geo.handler.GRPCGeoContentHandler: GetGeoContent")
	defer span.Finish()

	result, err := h.service.GetCityContent(ctx, request.Slug)
	if err != nil {
		if xerrors.Is(err, repository.ErrContentNotFound) {
			return result, status.Errorf(codes.NotFound, "city '%s' is not found", request.Slug)
		}

		h.logError(err)

		return result, errInternalError
	}

	return result, err
}

func (h *GRPCGeoContentHandler) GetCityImage(ctx context.Context, request *contentv1.GetContentByIdRequest) (
	response *contentv1.CityImage,
	err error,
) {
	span, ctx := opentracing.StartSpanFromContext(ctx, "internal.geo.handler.GRPCGeoContentHandler: GetCityImage")
	defer span.Finish()

	result, err := h.service.GetCityImage(ctx, request.Id)
	if err != nil {
		if xerrors.Is(err, repository.ErrContentNotFound) {
			return result, status.Errorf(codes.NotFound, "image of city '%d' is not found", request.Id)
		}

		h.logError(err)

		return result, errInternalError
	}

	return result, err
}

func (h *GRPCGeoContentHandler) GetAttractionContent(ctx context.Context, request *contentv1.GetContentBySlugRequest) (
	response *contentv1.AttractionContent,
	err error,
) {
	span, ctx := opentracing.StartSpanFromContext(ctx, "internal.geo.handler.GRPCGeoContentHandler: GetAttractionContent")
	defer span.Finish()

	result, err := h.service.GetAttractionContent(ctx, request.Slug)
	if err != nil {
		if xerrors.Is(err, repository.ErrContentNotFound) {
			return result, status.Errorf(codes.NotFound, "attraction '%s' is not found", request.Slug)
		}

		h.logError(err)

		return result, errInternalError
	}

	return result, err
}

func (h *GRPCGeoContentHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		contentv1.RegisterGeoContentServiceServer(server, h)
	}
}
