package geo

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/app"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/db"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/repository"
	v1 "a.yandex-team.ru/travel/marketing/content/v1"
)

type Service struct {
	logger    log.Logger
	reader    repository.GeoContentReader
	dbCluster *db.Cluster
}

func NewService(l log.Logger, dbCluster *db.Cluster, s3 app.S3Config) *Service {
	return &Service{logger: l, dbCluster: dbCluster, reader: *repository.NewGeoContentReader(s3)}
}

func (s *Service) GetCityContent(ctx context.Context, slug string) (result *v1.CityContent, err error) {
	conn, err := s.dbCluster.GetReadableConnection()
	if err != nil {
		return nil, fmt.Errorf("get connection error: %w", err)
	}

	content, err := s.reader.GetCityContent(ctx, conn, slug)
	if err != nil {
		return nil, fmt.Errorf("get city by slug=%s error: %w", slug, err)
	}

	return MapCityContent(content), err
}

func (s *Service) GetCityImage(ctx context.Context, settlementID int32) (*v1.CityImage, error) {
	conn, err := s.dbCluster.GetReadableConnection()
	if err != nil {
		return nil, fmt.Errorf("get connection error: %w", err)
	}

	imageURL, err := s.reader.GetCityImage(ctx, conn, settlementID)
	if err != nil {
		return nil, fmt.Errorf("get city image by id=%d error: %w", settlementID, err)
	}

	return &v1.CityImage{ImageUrl: imageURL}, nil
}

func (s *Service) GetAttractionContent(ctx context.Context, slug string) (*v1.AttractionContent, error) {
	conn, err := s.dbCluster.GetReadableConnection()
	if err != nil {
		return nil, fmt.Errorf("get connection error: %w", err)
	}

	content, err := s.reader.GetAttractionContent(ctx, conn, slug)
	if err != nil {
		return nil, fmt.Errorf("get attraction by slug=%s error: %w", slug, err)
	}

	return MapAttractionContent(content), nil
}
