package handler

import (
	"context"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/landing"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/repository"
	contentv1 "a.yandex-team.ru/travel/marketing/content/v1"
)

type GRPCBlogContentHandler struct {
	service *landing.Service
	logger  log.Logger
}

func NewGRPCBlogContentHandler(l log.Logger, srv *landing.Service) *GRPCBlogContentHandler {
	return &GRPCBlogContentHandler{service: srv, logger: l}
}

var errInternalError = status.Error(codes.Internal, "landing handler internal error")

func (h *GRPCBlogContentHandler) logError(err error) {
	h.logger.Error("landing handler error", log.Error(err))
}

func (h *GRPCBlogContentHandler) GetLanding(ctx context.Context, request *contentv1.GetContentBySlugRequest) (
	response *contentv1.Landing,
	err error,
) {
	span, ctx := opentracing.StartSpanFromContext(ctx, "internal.blog.handler.GRPCBlogContentHandler: GetLanding")
	defer span.Finish()

	result, err := h.service.GetLanding(ctx, request.Slug)
	if err != nil {
		if xerrors.Is(err, repository.ErrContentNotFound) {
			return result, status.Errorf(codes.NotFound, "landing '%s' is not found", request.Slug)
		}

		h.logError(err)

		return result, errInternalError
	}

	return result, err
}

func (h *GRPCBlogContentHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		contentv1.RegisterLandingContentServiceServer(server, h)
	}
}
