package db

import (
	"context"
	"fmt"
	"strings"

	"github.com/jackc/pgconn"
	"github.com/jackc/pgx/v4"
	"github.com/jackc/pgx/v4/log/zapadapter"
	"github.com/jackc/pgx/v4/stdlib"
	"golang.yandex/hasql"
	"golang.yandex/hasql/checkers"

	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/app"
)

type Cluster struct {
	cluster *hasql.Cluster
}

func New(config *app.DBConfig, logger *zap.Logger) (*Cluster, error) {
	var nodes []hasql.Node
	for _, hostname := range strings.Split(config.Hosts, ",") {
		connectionString := fmt.Sprintf(
			"host=%s port=%d dbname=%s user=%s password=%s",
			hostname,
			config.Port,
			config.DataBase,
			config.User,
			config.Password,
		)

		connectionConfig, err := pgx.ParseConfig(connectionString)
		if err != nil {
			return nil, fmt.Errorf("internal.pkg.db.NewCluster: %w", err)
		}

		connectionConfig.Logger = zapadapter.NewLogger(logger.L)
		connectionConfig.PreferSimpleProtocol = true
		connectionConfig.BuildStatementCache = nil
		connectionConfig.AfterConnect = func(ctx context.Context, conn *pgconn.PgConn) error {
			logger.Info("Connection to database established")
			return nil
		}

		db := stdlib.OpenDB(*connectionConfig)
		db.SetMaxOpenConns(config.MaxOpenConns)

		nodes = append(nodes, hasql.NewNode(hostname, db))
	}

	cluster, err := hasql.NewCluster(nodes, checkers.PostgreSQL, hasql.WithNodePicker(hasql.PickNodeClosest()))
	if err != nil {
		return nil, fmt.Errorf("could not create cluster: %w", err)
	}

	return &Cluster{cluster: cluster}, nil
}

func (c *Cluster) GetReadableConnection() (*Connection, error) {
	node := c.cluster.StandbyPreferred()
	if node == nil {
		return nil, fmt.Errorf("no PG connection available")
	}

	return &Connection{db: node.DB()}, nil
}
