package repository

import (
	"context"
	"database/sql"
	"fmt"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/app"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/db"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/model"
)

type GeoContentReader struct {
	cityMetaReader       ContentMetaReader
	attractionMetaReader ContentMetaReader
	s3                   app.S3Config
}

func NewGeoContentReader(s3 app.S3Config) *GeoContentReader {
	return &GeoContentReader{
		cityMetaReader:       *NewContentMetaReader(cityTableName, s3),
		attractionMetaReader: *NewContentMetaReader(attractionTableName, s3),
		s3:                   s3,
	}
}

const getCityContentQuery = `
		select id, description, info_box
		from %s
		where slug=$1;
	`

const getAttractionContentQuery = `
		select city_id, description, majority, preview_text
		from %s
		where slug=$1;
	`

const getCityChildrenQuery = `
		select name, slug
		from %s
		where city_id=$1
	`

const getAttractionParentQuery = `
		select name, slug
		from %s
		where id=$1
	`

const getCityImageQuery = `
		select avatars_image
		from %s
		where settlement_id=$1
	`

var cityContentRoot = model.NavigationNode{Slug: "city", Name: ""}

func (r GeoContentReader) prepareCityQuery(query string) string {
	return fmt.Sprintf(query, cityTableName)
}

func (r GeoContentReader) prepareAttractionQuery(query string) string {
	return fmt.Sprintf(query, attractionTableName)
}

func (r GeoContentReader) GetCityContent(ctx context.Context, conn *db.Connection, slug string) (city *model.City, err error) {
	meta, err := r.cityMetaReader.GetContentMeta(ctx, conn, slug)
	if err != nil {
		return nil, fmt.Errorf("get meta error: %w", err)
	}

	query := r.prepareCityQuery(getCityContentQuery)

	row := conn.QueryRow(ctx, query, slug)
	city = new(model.City)
	city.Meta = *meta
	var id int32

	err = row.Scan(
		&id,
		&city.Description,
		&city.InfoBox,
	)

	if err != nil {
		return nil, fmt.Errorf("get city content error: %w", err)
	}

	children, err := r.getChildren(ctx, conn, id)
	if err != nil {
		return nil, fmt.Errorf("get city attractions error: %w", err)
	}

	city.Navigation = model.Navigation{Children: children, Parent: cityContentRoot}

	return city, nil
}

func (r GeoContentReader) GetAttractionContent(ctx context.Context, conn *db.Connection, slug string) (attraction *model.Attraction, err error) {
	meta, err := r.attractionMetaReader.GetContentMeta(ctx, conn, slug)
	if err != nil {
		return nil, fmt.Errorf("get meta error: %w", err)
	}

	query := r.prepareAttractionQuery(getAttractionContentQuery)
	row := conn.QueryRow(ctx, query, slug)
	attraction = new(model.Attraction)
	attraction.Meta = *meta
	var cityID int32

	err = row.Scan(
		&cityID,
		&attraction.Description,
		&attraction.Majority,
		&attraction.PreviewText,
	)

	if err != nil {
		return nil, fmt.Errorf("get attraction content error: %w", err)
	}

	parent, err := r.readParent(ctx, conn, cityID)
	if err != nil {
		return nil, fmt.Errorf("get city of attraction error: %w", err)
	}

	attraction.Navigation = model.Navigation{Children: nil, Parent: *parent}

	return attraction, nil
}

func (r GeoContentReader) getChildren(ctx context.Context, conn *db.Connection, contentID int32) ([]model.NavigationNode, error) {
	query := r.prepareAttractionQuery(getCityChildrenQuery)
	rows, err := conn.Query(ctx, query, contentID)
	if err != nil {
		return nil, fmt.Errorf("get city children error: %w", err)
	}

	var children []model.NavigationNode
	for rows.Next() {
		node := new(model.NavigationNode)
		err := rows.Scan(
			&node.Name,
			&node.Slug,
		)

		if err != nil {
			return nil, fmt.Errorf("get city child node error: %w", err)
		}

		children = append(children, *node)
	}

	return children, nil
}

func (r GeoContentReader) readParent(ctx context.Context, conn *db.Connection, contentID int32) (*model.NavigationNode, error) {
	query := r.prepareCityQuery(getAttractionParentQuery)
	row := conn.QueryRow(ctx, query, contentID)
	node := new(model.NavigationNode)
	err := row.Scan(
		&node.Name,
		&node.Slug,
	)

	if err != nil {
		return nil, fmt.Errorf("get attraction parent node error: %w", err)
	}

	return node, nil
}

func (r GeoContentReader) GetCityImage(ctx context.Context, conn *db.Connection, settlementID int32) (string, error) {
	query := r.prepareCityQuery(getCityImageQuery)
	var image string
	row := conn.QueryRow(ctx, query, settlementID)
	err := row.Scan(&image)

	if err != nil {
		if xerrors.Is(err, sql.ErrNoRows) {
			return "", ErrContentNotFound
		}

		return "", fmt.Errorf("get city image query error: %w", err)
	}

	return image, nil
}
