package repository

import (
	"context"
	"database/sql"
	"fmt"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/app"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/db"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/model"
)

type ContentMetaReader struct {
	tableName ContentTableName
	s3config  app.S3Config
}

func NewContentMetaReader(tableName ContentTableName, s3config app.S3Config) *ContentMetaReader {
	return &ContentMetaReader{tableName: tableName, s3config: s3config}
}

const getMetaQuery = `
		select
			name,
			slug,
			title,
			seo_description,
			image,
			og_title,
			og_description,
			og_image,
			twitter_title,
			twitter_description,
			twitter_image
		from %s
		where slug = $1;
	`

func (r ContentMetaReader) GetContentMeta(ctx context.Context, conn *db.Connection, slug string) (*model.ContentMeta, error) {
	query := fmt.Sprintf(getMetaQuery, r.tableName)
	row := conn.QueryRow(ctx, query, slug)
	meta := new(model.ContentMeta)

	err := row.Scan(
		&meta.Name,
		&meta.Slug,
		&meta.Title,
		&meta.SeoDescription,
		&meta.Image,
		&meta.OGTitle,
		&meta.OGDescription,
		&meta.OGImage,
		&meta.TwitterTitle,
		&meta.TwitterDescription,
		&meta.TwitterImage,
	)

	if err != nil {
		if xerrors.Is(err, sql.ErrNoRows) {
			return nil, ErrContentNotFound
		}

		return nil, fmt.Errorf("can not get content meta: %w", err)
	}

	meta.Image = MakeImagePath(meta.Image, r.s3config)
	meta.OGImage = MakeImagePath(meta.OGImage, r.s3config)
	meta.TwitterImage = MakeImagePath(meta.TwitterImage, r.s3config)

	return meta, nil
}
