package repository

import (
	"context"
	"database/sql"
	"fmt"

	"a.yandex-team.ru/travel/marketing/content/backend/internal/app"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/db"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/model"
)

type SeoContentReader struct {
	metaReader ContentMetaReader
}

func NewSeoContentReader(s3 app.S3Config) *SeoContentReader {
	return &SeoContentReader{metaReader: *NewContentMetaReader(seoTableName, s3)}
}

const getSeoContentQuery = `
		select id, description, use_navigation, parent_id
		from %s
		where slug = $1;
	`

const getSeoChildrenQuery = `
		select name, slug
		from %s
		where parent_id=$1
	`

const getSeoParentQuery = `
		select name, slug
		from %s
		where id=$1
`

var seoContentRoot = model.NavigationNode{Slug: "seo", Name: ""}

func (r SeoContentReader) prepareQuery(query string) string {
	return fmt.Sprintf(query, seoTableName)
}

func (r SeoContentReader) GetSeoContent(ctx context.Context, conn *db.Connection, slug string) (seoPage *model.SeoContent, err error) {
	meta, err := r.metaReader.GetContentMeta(ctx, conn, slug)
	if err != nil {
		return nil, fmt.Errorf("get meta error: %w", err)
	}

	query := r.prepareQuery(getSeoContentQuery)
	row := conn.QueryRow(ctx, query, slug)
	seoPage = new(model.SeoContent)
	seoPage.Meta = *meta
	var id int32
	var parentID sql.NullInt32

	err = row.Scan(
		&id,
		&seoPage.Description,
		&seoPage.UseNavigation,
		&parentID,
	)

	if err != nil {
		return nil, fmt.Errorf("seo content query error: %w", err)
	}

	children, err := r.getChildren(ctx, conn, id)
	if err != nil {
		return nil, fmt.Errorf("get seo navigation error: %w", err)
	}

	parent, err := r.getParent(ctx, conn, parentID)
	if err != nil {
		return nil, fmt.Errorf("get seo navigation error: %w", err)
	}

	seoPage.Navigation = model.Navigation{Children: children, Parent: *parent}

	return seoPage, nil
}

func (r SeoContentReader) getChildren(ctx context.Context, conn *db.Connection, contentID int32) ([]model.NavigationNode, error) {
	rows, err := conn.Query(ctx, r.prepareQuery(getSeoChildrenQuery), contentID)
	if err != nil {
		return nil, fmt.Errorf("get children error: %w", err)
	}

	var children []model.NavigationNode
	for rows.Next() {
		node := new(model.NavigationNode)
		err := rows.Scan(
			&node.Name,
			&node.Slug,
		)

		if err != nil {
			return nil, fmt.Errorf("get navigation child node error: %w", err)
		}

		children = append(children, *node)
	}

	return children, nil
}

func (r SeoContentReader) getParent(ctx context.Context, conn *db.Connection, parentID sql.NullInt32) (*model.NavigationNode, error) {
	if parentID.Valid {
		row := conn.QueryRow(ctx, r.prepareQuery(getSeoParentQuery), parentID)
		parent, err := r.readNode(row)
		if err != nil {
			return nil, fmt.Errorf("get parent node error: %w", err)
		}

		return parent, nil
	}

	return &seoContentRoot, nil
}

func (r SeoContentReader) readNode(row *sql.Row) (*model.NavigationNode, error) {
	node := new(model.NavigationNode)
	err := row.Scan(
		&node.Name,
		&node.Slug,
	)

	if err != nil {
		return nil, fmt.Errorf("get node error: %w", err)
	}

	return node, nil
}
