package handler

import (
	"context"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/pkg/repository"
	"a.yandex-team.ru/travel/marketing/content/backend/internal/seo"
	contentv1 "a.yandex-team.ru/travel/marketing/content/v1"
)

type GRPCSeoContentHandler struct {
	service *seo.Service
	logger  log.Logger
}

func NewGRPCSeoContentHandler(l log.Logger, srv *seo.Service) *GRPCSeoContentHandler {
	return &GRPCSeoContentHandler{service: srv, logger: l}
}

var errInternalError = status.Error(codes.Internal, "landing handler internal error")

func (h *GRPCSeoContentHandler) logError(err error) {
	h.logger.Error("geo handler error", log.Error(err))
}

func (h *GRPCSeoContentHandler) GetSeoContent(ctx context.Context, request *contentv1.GetContentBySlugRequest) (
	response *contentv1.SeoContent,
	err error,
) {
	span, ctx := opentracing.StartSpanFromContext(ctx, "internal.seo.handler.GRPCSeoContentHandler: GetSeoContent")
	defer span.Finish()

	result, err := h.service.GetSeoContent(ctx, request.Slug)
	if err != nil {
		if xerrors.Is(err, repository.ErrContentNotFound) {
			return result, status.Errorf(codes.NotFound, "city '%s' is not found", request.Slug)
		}

		h.logError(err)

		return result, errInternalError
	}

	return result, err
}

func (h *GRPCSeoContentHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		contentv1.RegisterSeoContentServiceServer(server, h)
	}
}
