package main

import (
	"context"
	"fmt"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/travel/library/go/configuration"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	traveltvm "a.yandex-team.ru/travel/library/go/tvm"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/db"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/repository"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/service/s3"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/tvmapi"
	storyHandler "a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/story/handler"
	uploadHandler "a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/upload/handler"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/uploader"
)

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	configLoader := configuration.NewDefaultConfitaLoader()
	err := configLoader.Load(ctx, &configs.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.New(&configs.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer("travel.folk_guide_contest")

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
		ctxCancel()
	}()

	var (
		tvmClient     tvm.Client
		tvmAllowedIds []tvm.ClientID
	)
	if configs.Cfg.Tvm.Enabled {
		tvmAllowedIds = traveltvm.TvmClientIDFromInt(configs.Cfg.Tvm.WhiteList)
		tvmClient, err = tvmapi.NewAPIClient(configs.Cfg.Tvm, logger)

		if err != nil {
			fmt.Println("failed to create tvm client: ", err)
			return
		} else {
			logger.Infof("configured tvm client (whitelist=%d)", configs.Cfg.Tvm.WhiteList)
		}
	}

	rootRegistry := metrics.NewRegistryWithQloudTags()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, configs.Cfg.Metrics.PerfMetricsRefreshInterval)

	pgpool, err := db.NewPoolWithRetries(ctx, &configs.Cfg.PG, logger)
	if err != nil {
		fmt.Println("can not connect to postgres cluster, err:", err)
		return
	}

	storyRepository := repository.NewStoryRepository()
	uploadRepository := repository.NewUploadRepository()

	// configure uploader
	s3client, err := s3.NewClient(&configs.Cfg.S3, logger)
	if err != nil {
		logger.Errorf("can not configure s3 client: %v", err)
		return
	}

	s3uploader := uploader.NewUploader(
		&configs.Cfg.S3,
		s3client,
		uploadRepository,
		pgpool,
		logger,
	)

	logger.Info("run_uploading")
	go s3uploader.RunUploading(ctx)

	// bootstrapping http api
	httpStoriesHandler := storyHandler.NewHTTPStoriesHandler(
		&configs.Cfg,
		pgpool,
		storyRepository,
		logger,
	)

	httpAPIRouteBuilders := []httpserver.RouteBuilder{
		httpStoriesHandler.GetRouteBuilder(),
	}

	// bootstrapping hooks api
	httpUploadHandler := uploadHandler.NewHTTPHooksHandler(
		&configs.Cfg,
		pgpool,
		storyRepository,
		uploadRepository,
		logger,
	)
	httpHooksRouteBuilders := []httpserver.RouteBuilder{
		httpUploadHandler.GetRouteBuilder(),
	}

	// running server
	wg := sync.WaitGroup{}

	wg.Add(1)
	go func() {
		defer wg.Done()
		err = httpserver.RunHTTPServer(
			context.Background(),
			configs.Cfg.HTTP,
			httpAPIRouteBuilders,
			logger,
			tvmClient,
			tvmAllowedIds,
			rootRegistry.WithPrefix("http"),
		)
		if err != nil {
			logger.Fatal("Error while starting HTTP server", log.Error(err))
		}
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		err = httpserver.RunHTTPServer(context.Background(),
			httpserver.HTTPConfig{Addr: configs.Cfg.HooksHTTP.Addr},
			httpHooksRouteBuilders,
			logger,
			nil,
			nil,
			rootRegistry.WithPrefix("http-hooks"),
		)
		if err != nil {
			logger.Fatal("Error while starting HTTP hooks server", log.Error(err))
		}
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		err = metricserver.RunMetricsHTTPServer(context.Background(), configs.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()
	logger.Info("Started", log.String("addr", configs.Cfg.HTTP.Addr))
	wg.Wait()
	logger.Info("End")
}
