package main

import (
	"context"
	"fmt"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/mailing"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/db"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/repository"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/service/sender"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/service/startrek"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/tracker"
)

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &configs.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.New(&configs.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	rootRegistry := metrics.NewRegistryWithQloudTags()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, configs.Cfg.Metrics.PerfMetricsRefreshInterval)

	pgpool, err := db.NewPoolWithRetries(ctx, &configs.Cfg.PG, logger)
	if err != nil {
		fmt.Println("can not connect to postgres cluster, err:", err)
		return
	}

	startrekController := startrek.NewController(logger, &configs.Cfg.Startrek)
	senderClient := sender.NewHTTPClient(logger, &configs.Cfg.Sender)

	storyRepository := repository.NewStoryRepository()

	mailingController := mailing.NewController(
		logger,
		pgpool,
		startrekController,
		senderClient,
		storyRepository,
	)

	trackerController := tracker.NewController(
		logger,
		pgpool,
		&configs.Cfg.S3,
		startrekController,
		storyRepository,
	)

	wg := sync.WaitGroup{}

	wg.Add(1)
	go func() {
		defer wg.Done()
		mailingController.Run(ctx)
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		trackerController.Run(ctx)
	}()

	// running metrics server
	wg.Add(1)
	go func() {
		defer wg.Done()

		err = metricserver.RunMetricsHTTPServer(
			context.Background(),
			metricserver.MetricsConfig(configs.Cfg.MailingMetrics),
			logger, rootRegistry,
		)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()
	logger.Structured().Info("Started metrics", log.String("addr", configs.Cfg.MailingMetrics.Addr))
	wg.Wait()
}
