package mailing

import (
	"context"
	"errors"
	"github.com/jackc/pgx/v4/pgxpool"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/metrics"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/model"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/repository"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/service/startrek"
)

type SenderClient interface {
	SendConfirmed(mail string) ([]byte, error)
	SendRejected(mail string) ([]byte, error)
}

type Controller struct {
	logger             log.Logger
	pool               *pgxpool.Pool
	startrekController *startrek.Controller
	senderClient       SenderClient
	storyRepository    *repository.StoryRepository
}

func NewController(
	logger log.Logger,
	pool *pgxpool.Pool,
	startrekController *startrek.Controller,
	senderClient SenderClient,
	storyRepository *repository.StoryRepository,
) *Controller {
	return &Controller{
		logger:             logger,
		pool:               pool,
		startrekController: startrekController,
		senderClient:       senderClient,
		storyRepository:    storyRepository,
	}
}

func (c Controller) Run(ctx context.Context) {
	for {
		select {
		case <-time.After(time.Hour):
			c.runOnce(ctx)
		case <-ctx.Done():
			c.logger.Structured().Info("finish mailing cycle")
			return
		}
	}
}

func (c Controller) runOnce(ctx context.Context) {
	var err error

	c.logger.Structured().Info("run tracker sync")
	err = c.syncReadyIssuesWithDB(ctx)
	if err != nil {
		c.logger.Structured().Error("tracker syncing fail", log.Error(err))
		metrics.MailingRegistry.AddError()
	}

	c.logger.Structured().Info("run mailing")
	err = c.sendMails(ctx)
	if err != nil {
		c.logger.Structured().Error("mailing fail", log.Error(err))
		metrics.MailingRegistry.AddError()
	}
}

func (c *Controller) syncReadyIssuesWithDB(ctx context.Context) error {
	issues, err := c.startrekController.GetMailingIssues()
	if err != nil {
		return errorSyncReadyIssuesWithDB.Wrap(err)
	}

	var unresolved int
	for _, issue := range issues {
		err = c.syncIssue(ctx, issue)
		if errors.Is(err, errorUnresolvedIssue) {
			unresolved++
			continue
		}

		if err != nil {
			c.logger.Structured().Error("occurred an error while syncing story", log.Error(err))
			metrics.MailingRegistry.AddError()
		} else {
			c.logger.Structured().Info("synced story", log.String("storyId", issue.Unique))
		}
		time.Sleep(time.Second)
	}
	metrics.MailingRegistry.SetUnresolvedIssues(unresolved)
	return nil
}

func (c Controller) syncIssue(ctx context.Context, issue startrek.Issue) error {
	story, err := c.storyRepository.GetStory(ctx, c.pool, issue.Unique)
	if err != nil {
		return errorSyncIssue.Wrap(err)
	}

	if story == nil {
		return errorNotFoundStoryKey
	}

	if !startrek.IsResolvedIssue(issue) {
		return errorUnresolvedIssue
	}

	story.MailingState = model.MailingReady
	switch issue.Resolution.Key {
	case startrek.SuccessfulResolution:
		story.Resolution = model.ResolutionApproved
	case startrek.DontDoResolution:
		story.Resolution = model.ResolutionRejected
	default:
		c.logger.Structured().Error(
			"unexpected issue resolution",
			log.String("storyId", issue.Unique),
			log.String("resolution", string(issue.Resolution.Key)),
		)
		return errorSyncIssue.Wrap(errorUnexpectedResolution)
	}

	tx, err := c.pool.Begin(ctx)
	if err != nil {
		return errorSyncIssue.Wrap(err)
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		}
	}()

	err = c.storyRepository.UpdateStory(ctx, tx, story)
	if err != nil {
		return errorSyncIssue.Wrap(err)
	}

	err = c.removeMailingTag(issue)
	if err != nil {
		return errorSyncIssue.Wrap(err)
	}

	if err := tx.Commit(ctx); err != nil {
		return errorSyncIssue.Wrap(err)
	}

	return nil
}

func (c Controller) removeMailingTag(issue startrek.Issue) error {
	err := c.startrekController.RemoveMailingTag(issue)
	if err != nil {
		return errorRemoveMailingTag.Wrap(err)
	}
	return nil
}

func (c Controller) sendMails(ctx context.Context) error {
	stories, err := c.storyRepository.GetStoriesForMailing(context.Background(), c.pool)

	if err != nil {
		return errorSendMails.Wrap(err)
	}

	for _, story := range stories {
		err = c.sendMail(ctx, story)
		if err != nil {
			c.logger.Structured().Error("occurred an error while sending mail", log.Error(err))
			metrics.MailingRegistry.AddError()
		} else {
			c.logger.Structured().Info("mail sent", log.String("storyId", story.UID))
		}
		time.Sleep(time.Second)
	}
	return nil
}

func (c Controller) sendMail(ctx context.Context, story *model.Story) error {
	story.MailingState = model.MailingStart
	err := c.updateStory(ctx, story)
	if err != nil {
		return errorSendMail.Wrap(err)
	}

	tx, err := c.pool.Begin(ctx)
	if err != nil {
		return errorUpdateStory.Wrap(err)
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		}
	}()

	switch story.Resolution {
	case model.ResolutionApproved:
		_, err = c.senderClient.SendConfirmed(story.Email)
	case model.ResolutionRejected:
		_, err = c.senderClient.SendRejected(story.Email)
	default:
		c.logger.Structured().Error(
			"unexpected story resolution",
			log.String("storyId", story.UID),
			log.Int("resolution", int(story.Resolution)),
		)
		return errorSendMail.Wrap(errorUnexpectedResolution)
	}

	if err != nil {
		return errorSendMail.Wrap(err)
	}

	story.MailingState = model.MailingHandled
	err = c.storyRepository.UpdateStory(ctx, tx, story)
	if err != nil {
		return errorSendMail.Wrap(err)
	}
	if err := tx.Commit(ctx); err != nil {
		return errorSendMail.Wrap(err)
	}
	return nil
}

func (c Controller) updateStory(ctx context.Context, story *model.Story) error {
	tx, err := c.pool.Begin(ctx)
	if err != nil {
		return errorUpdateStory.Wrap(err)
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		}
	}()

	err = c.storyRepository.UpdateStory(ctx, tx, story)
	if err != nil {
		return errorUpdateStory.Wrap(err)
	}

	if err := tx.Commit(ctx); err != nil {
		return errorUpdateStory.Wrap(err)
	}
	return nil
}
