package db

import (
	"context"
	"fmt"

	"github.com/jackc/pgx/v4"
	"github.com/jackc/pgx/v4/log/zapadapter"
	"github.com/jackc/pgx/v4/pgxpool"

	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
)

func NewPool(ctx context.Context, config *configs.DB, logger *zap.Logger) (*pgxpool.Pool, error) {
	logLevel := logLevelFromEnvOrDefault(pgx.LogLevelError, config)

	connString := fmt.Sprintf(
		"host=%s port=%d dbname=%s user=%s password=%s sslmode=verify-full pool_max_conns=20",
		config.Host,
		config.Port,
		config.DataBase,
		config.User,
		config.Password,
	)
	poolConfig, err := pgxpool.ParseConfig(connString)

	if err != nil {
		return nil, fmt.Errorf("internal.pkg.db.NewPool: %w", err)
	}

	poolConfig.ConnConfig.Logger = zapadapter.NewLogger(logger.L)
	poolConfig.ConnConfig.LogLevel = logLevel
	poolConfig.ConnConfig.PreferSimpleProtocol = true
	poolConfig.AfterConnect = func(ctx context.Context, conn *pgx.Conn) error {
		logger.Info("Connection to database established")
		return nil
	}

	logger.Info("Connecting to database pool")

	pool, err := pgxpool.ConnectConfig(ctx, poolConfig)
	if err != nil {
		return nil, fmt.Errorf("internal.pkg.db.NewPool: %w", err)
	}
	return pool, nil
}

func logLevelFromEnvOrDefault(defaultLogLevel pgx.LogLevel, config *configs.DB) pgx.LogLevel {
	if len(config.LogLevel) == 0 {
		return defaultLogLevel
	}
	ll, err := pgx.LogLevelFromString(config.LogLevel)
	if err != nil {
		return defaultLogLevel
	}
	return ll
}
