package repository

import (
	"context"
	"database/sql"
	"fmt"
	"time"

	"github.com/gofrs/uuid"
	"github.com/jackc/pgtype/pgxtype"
	"github.com/jackc/pgx/v4"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/model"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/service/s3"
)

type StoryRepository struct{}

func NewStoryRepository() *StoryRepository {
	return &StoryRepository{}
}

const (
	InsertStoryQuery = `
		insert into story (
		   id,
		   yandex_uid,
		   state,
		   title,
		   email,
		   first_name,
		   last_name,
		   story_type,
		   created_at,
		   updated_at,
		   ticket
		) values ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11);
	`
	GetStoryQuery = `
		select
			id,
			yandex_uid,
			state,
			story_type,
			title,
			email,
			first_name,
			last_name,
			resolution,
			mailing_state,
			created_at,
			updated_at,
			ticket
		from story
		where id = $1;
	`
	GetStoriesForMailingQuery = `
		select
			id,
			yandex_uid,
			state,
			story_type,
			title,
			email,
			first_name,
			last_name,
			resolution,
			mailing_state,
			created_at,
			updated_at,
			ticket
		from story
		where mailing_state = $1;
	`

	GetStoriesByState = `
		select
			id,
			yandex_uid,
			state,
			story_type,
			title,
			email,
			first_name,
			last_name,
			resolution,
			mailing_state,
			created_at,
			updated_at,
			ticket
		from story
		where state = $1;
	`

	GetStoryBlocks = `
		SELECT
			sb.id,
			sb.story_id,
			sb.text,
			su.upload_id
		FROM story_block sb
		LEFT JOIN story_upload su 
			ON su.story_block_id = sb.id
		WHERE sb.story_id = $1
		ORDER BY sb.block_no ASC;
	`

	SaveStoryQuery = `
		update story
		set state = $3,
			title = $4,
			email = $5,
			first_name = $6,
			last_name = $7,
			story_type = $8,
			updated_at = $9,
			ticket = $10,
			mailing_state = $11,
			resolution = $12
		where id=$1 and state=$2;
	`

	InsertBlockQuery = `
		insert into story_block (story_id, text, block_no)
		values ($1, $2, $3) returning id
	`

	InsertBlockUploadQuery = `
		insert into story_upload (story_block_id, upload_no, upload_id)
		values ($1, $2, $3)
	`

	UpdateStoryState = `
		UPDATE story
		SET state = $2
		WHERE id=$1;
	`
)

func (r StoryRepository) NewStory(ctx context.Context, conn pgxtype.Querier, YandexUID string) (*model.Story, error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, tracingNewStory)
	defer span.Finish()

	uid, err := uuid.NewV4()
	if err != nil {
		return nil, errorNewStory.Wrap(err)
	}

	now := time.Now()
	story := model.Story{
		UID:       uid.String(),
		YandexUID: YandexUID,
		CreatedAt: now,
		UpdatedAt: now,
	}

	_, err = conn.Exec(
		ctx, InsertStoryQuery,
		story.UID,
		story.YandexUID,
		story.State,
		story.Title,
		story.Email,
		story.FirstName,
		story.LastName,
		story.Type,
		story.CreatedAt,
		story.UpdatedAt,
		story.Ticket,
	)

	if err != nil {
		return nil, errorNewStory.Wrap(err)
	}

	return &story, nil
}

func (r StoryRepository) GetStory(ctx context.Context, conn pgxtype.Querier, storyUID string) (*model.Story, error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, tracingGetStory)
	defer span.Finish()

	row := conn.QueryRow(ctx, GetStoryQuery, storyUID)
	story := new(model.Story)

	err := row.Scan(
		&story.UID,
		&story.YandexUID,
		&story.State,
		&story.Type,
		&story.Title,
		&story.Email,
		&story.FirstName,
		&story.LastName,
		&story.Resolution,
		&story.MailingState,
		&story.CreatedAt,
		&story.UpdatedAt,
		&story.Ticket,
	)
	if err != nil {
		if xerrors.Is(err, pgx.ErrNoRows) {
			return nil, nil
		}
		return nil, errorGetStory.Wrap(err)
	}

	return story, nil
}

func (r StoryRepository) GetStoriesForMailing(ctx context.Context, conn pgxtype.Querier) ([]*model.Story, error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, tracingGetStoriesForMailing)
	defer span.Finish()

	rows, err := conn.Query(ctx, GetStoriesForMailingQuery, model.MailingReady)
	if err != nil {
		return nil, errorGetStory.Wrap(err)
	}
	return r.buildStories(rows)
}

func (r StoryRepository) GetStoriesWithoutIssues(ctx context.Context, conn pgxtype.Querier) ([]*model.Story, error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, tracingGetStoriesWithoutIssues)
	defer span.Finish()

	rows, err := conn.Query(ctx, GetStoriesByState, model.Saved)
	if err != nil {
		return nil, errorGetStory.Wrap(err)
	}
	return r.buildStories(rows)
}

func (r StoryRepository) buildStories(rows pgx.Rows) ([]*model.Story, error) {
	var result []*model.Story
	for rows.Next() {
		story := new(model.Story)

		err := rows.Scan(
			&story.UID,
			&story.YandexUID,
			&story.State,
			&story.Type,
			&story.Title,
			&story.Email,
			&story.FirstName,
			&story.LastName,
			&story.Resolution,
			&story.MailingState,
			&story.CreatedAt,
			&story.UpdatedAt,
			&story.Ticket,
		)
		if err != nil {
			return nil, errorGetStory.Wrap(err)
		}

		result = append(result, story)
	}
	return result, nil
}

func (r StoryRepository) UpdateStory(ctx context.Context, tx pgx.Tx, story *model.Story) error {
	span, ctx := opentracing.StartSpanFromContext(ctx, tracingUpdateStory)
	defer span.Finish()

	now := time.Now()

	commandTag, err := tx.Exec(
		ctx, SaveStoryQuery,
		story.UID,
		story.State,
		model.Saved,
		story.Title,
		story.Email,
		story.FirstName,
		story.LastName,
		story.Type,
		now,
		story.Ticket,
		story.MailingState,
		story.Resolution,
	)
	if err != nil {
		return errorSendStory.Wrap(err)
	}

	if commandTag.RowsAffected() == 0 {
		return fmt.Errorf("no registered new story found by storyId (%s)", story.UID)
	}

	for blockNo, block := range story.Blocks {
		row := tx.QueryRow(
			ctx, InsertBlockQuery,
			block.StoryUID,
			block.Text,
			blockNo,
		)

		err = row.Scan(&block.ID)
		if err != nil {
			return errorSendStory.Wrap(err)
		}

		for uploadNo, upload := range block.Uploads {
			_, err = tx.Exec(
				ctx, InsertBlockUploadQuery,
				block.ID,
				uploadNo,
				upload.UploadUID,
			)

			if err != nil {
				return errorSendStory.Wrap(err)
			}
		}
	}
	return nil
}

func (r StoryRepository) UpdateStoryState(ctx context.Context, tx pgx.Tx, story *model.Story, state model.StoryState) error {
	span, ctx := opentracing.StartSpanFromContext(ctx, tracingUpdateStoryState)
	defer span.Finish()

	commandTag, err := tx.Exec(ctx, UpdateStoryState, story.UID, state)

	if err != nil {
		return errorUpdateStorySTate.Wrap(err)
	}

	if commandTag.RowsAffected() == 0 {
		return fmt.Errorf("no registered new story found by storyId (%s)", story.UID)
	}
	return nil
}

func (r StoryRepository) PopulateStoryWithBlocks(ctx context.Context, conn pgxtype.Querier, story *model.Story, s3conf *configs.S3) error {
	span, ctx := opentracing.StartSpanFromContext(ctx, tracingPopulateStoryWithBlocks)
	defer span.Finish()

	rows, err := conn.Query(ctx, GetStoryBlocks, story.UID)
	if err != nil {
		return errorPopulateStoryWithBlocks.Wrap(err)
	}

	uploadsByBlockIDs := make(map[uint][]model.StoryBlockUpload)
	blocksByIds := make(map[uint]*model.StoryBlock)
	for rows.Next() {
		storyBlock := new(model.StoryBlock)
		uploadBlock := new(model.StoryBlockUpload)
		var uploadUID sql.NullString
		err := rows.Scan(
			&storyBlock.ID,
			&storyBlock.StoryUID,
			&storyBlock.Text,
			&uploadUID,
		)
		if err != nil {
			return errorPopulateStoryWithBlocks.Wrap(err)
		}

		if _, found := blocksByIds[storyBlock.ID]; !found {
			blocksByIds[storyBlock.ID] = storyBlock
		}
		if uploadUID.Valid {
			if _, found := uploadsByBlockIDs[storyBlock.ID]; !found {
				uploadsByBlockIDs[storyBlock.ID] = make([]model.StoryBlockUpload, 0)
			}
			uploadBlock.UploadUID = uploadUID.String
			uploadBlock.PublicURL = s3.MakeURL(uploadBlock.UploadUID, s3conf)
			uploadsByBlockIDs[storyBlock.ID] = append(uploadsByBlockIDs[storyBlock.ID], *uploadBlock)
		}
	}

	blocks := make([]model.StoryBlock, 0)
	for blockID, block := range blocksByIds {
		block.Uploads = uploadsByBlockIDs[blockID]
		blocks = append(blocks, *block)
	}
	story.Blocks = blocks
	return nil
}
