package repository

import (
	"context"
	"time"

	"github.com/jackc/pgtype/pgxtype"
	"github.com/jackc/pgx/v4"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/model"
)

type UploadRepository struct{}

func NewUploadRepository() *UploadRepository {
	return &UploadRepository{}
}

const (
	SelectUploadByTusIDQuery = `
		select
			id,
			story_id,
			file_name,
			file_type,
			status,
			created_at
		from upload
		where tus_id = $1
	`
	InsertUploadQuery = `
		insert into upload (
		    id,
            tus_id,
		    story_id,
		    file_name,
		    file_type,
			status,
		    created_at
		) values ($1, $2, $3, $4, $5, $6, $7);
	`
	UpdateUploadQuery = `
		update upload
		set
			status = $2
		where id = $1
	`
)

func (r UploadRepository) NewUpload(
	ctx context.Context,
	conn pgxtype.Querier,
	uploadUID,
	tusID,
	storyUID,
	fileName,
	fileType string,
) (*model.Upload, error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, tracingNewUpload)
	defer span.Finish()

	now := time.Now()
	upload := model.Upload{
		UID:       uploadUID,
		TusID:     tusID,
		StoryUID:  storyUID,
		FileName:  fileName,
		FileType:  fileType,
		CreatedAt: now,
	}

	_, err := conn.Exec(
		ctx, InsertUploadQuery,
		upload.UID,
		upload.TusID,
		upload.StoryUID,
		upload.FileName,
		upload.FileType,
		upload.Status,
		now,
	)

	if err != nil {
		return nil, errorNewUpload.Wrap(err)
	}

	return &upload, nil
}

func (r UploadRepository) GetUploadByTusID(ctx context.Context, tx pgxtype.Querier, tusID string) (*model.Upload, error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, tracingGetUploadsByTusID)
	defer span.Finish()

	upload := &model.Upload{TusID: tusID}
	row := tx.QueryRow(ctx, SelectUploadByTusIDQuery, tusID)

	err := row.Scan(
		&upload.UID,
		&upload.StoryUID,
		&upload.FileName,
		&upload.FileType,
		&upload.Status,
		&upload.CreatedAt,
	)
	if err != nil {
		if xerrors.Is(err, pgx.ErrNoRows) {
			return nil, nil
		}
		return nil, errorGetUploadsByTusID.Wrap(err)
	}

	return upload, nil
}

func (r UploadRepository) UpdateUpload(ctx context.Context, conn pgxtype.Querier, upload *model.Upload) error {
	span, ctx := opentracing.StartSpanFromContext(ctx, tracingUpdateUpload)
	defer span.Finish()

	_, err := conn.Exec(ctx, UpdateUploadQuery, upload.UID, upload.Status)
	if err != nil {
		return errorUpdateUpload.Wrap(err)
	}
	return nil
}
