package s3

import (
	"context"
	"fmt"
	"io"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
)

type Client struct {
	config  *configs.S3
	session *session.Session
	client  *s3.S3
	logger  log.Logger
}

func NewClient(config *configs.S3, logger log.Logger) (*Client, error) {
	// Takes values from env vars AWS_ACCESS_KEY, AWS_SECRET_KEY. Ensured in Config
	awsSession, err := session.NewSession(&aws.Config{
		Endpoint:    aws.String(config.Endpoint),
		Region:      aws.String(config.Region),
		Credentials: credentials.NewEnvCredentials(),
		MaxRetries:  aws.Int(config.MaxRetries),
	})
	if err != nil {
		return nil, fmt.Errorf("pkg.service.s3.NewClient: %w", err)
	}
	return &Client{
		config:  config,
		session: awsSession,
		client:  s3.New(awsSession),
		logger:  logger,
	}, nil
}

func (c *Client) EnsureBucket() error {
	input := &s3.CreateBucketInput{
		Bucket: aws.String(c.config.Bucket),
	}

	_, err := c.client.CreateBucket(input)
	if err != nil {
		if aerr, ok := err.(awserr.Error); ok {
			switch aerr.Code() {
			case s3.ErrCodeBucketAlreadyExists, s3.ErrCodeBucketAlreadyOwnedByYou:
				return nil
			}
		}
		return fmt.Errorf("pkg.service.s3.client.EnsureBucket: %w", err)
	}

	return nil
}

func (c *Client) PutObject(ctx context.Context, key string, reader io.Reader, contentType string) error {
	key = makeKey(key, c.config)
	input := &s3.PutObjectInput{
		Key:         aws.String(key),
		Body:        aws.ReadSeekCloser(reader),
		Bucket:      aws.String(c.config.Bucket),
		ContentType: aws.String(contentType),
	}

	_, err := c.client.PutObjectWithContext(ctx, input)
	if err != nil {
		return fmt.Errorf("pkg.service.s3.client.PutObject: %w", err)
	}
	c.logger.Infof("stored key: %s", key)

	return nil
}
