package sender

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strings"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
)

const (
	errorClientSend      = "sender.client.Send: %w"
	errorClientDoRequest = "sender.client.doRequest: %w"

	contentTypeHeader = "Content-Type"
	contentTypeValue  = "application/x-www-form-urlencoded"
)

type HTTPClient struct {
	config     *configs.Sender
	httpClient *http.Client
	logger     log.Logger
}

func NewHTTPClient(logger log.Logger, config *configs.Sender) *HTTPClient {
	httpClient := &http.Client{
		Timeout: time.Duration(configs.Cfg.Sender.Timeout) * time.Millisecond,
	}
	return &HTTPClient{
		config:     config,
		httpClient: httpClient,
		logger:     logger,
	}
}

func (c HTTPClient) SendConfirmed(mail string) ([]byte, error) {
	return c.send(c.config.CampaignSlugConfirmed, mail)
}

func (c HTTPClient) SendRejected(mail string) ([]byte, error) {
	return c.send(c.config.CampaignSlugRejected, mail)
}

type responseRoot struct {
	Result responseResult `json:"result"`
}

type responseResult struct {
	Status string `json:"status"`
}

func (c *HTTPClient) send(campaignSlug, toEmail string) ([]byte, error) {
	var byteResponse []byte
	request := func() error {
		sendURL := c.getBaseURL(campaignSlug) + "send"

		body := c.getData(toEmail).Encode()

		req, err := http.NewRequest("POST", sendURL, strings.NewReader(body))
		if err != nil {
			return fmt.Errorf(errorClientSend, err)
		}
		req.Header.Set(contentTypeHeader, contentTypeValue)
		req.SetBasicAuth(c.config.AuthUser, "")

		response, err := c.httpClient.Do(req)
		if err != nil {
			return fmt.Errorf(errorClientSend, err)
		}
		defer response.Body.Close()

		byteResponse, err = ioutil.ReadAll(response.Body)
		if response.StatusCode > 400 {
			return fmt.Errorf("sender answered with code %d: %s", response.StatusCode, string(byteResponse))
		}

		if err != nil {
			return fmt.Errorf(errorClientSend, err)
		}

		if err = c.parseResponse(byteResponse); err != nil {
			return fmt.Errorf(errorClientSend, err)
		}
		return nil
	}
	err := c.retryRequest(request)
	if err != nil {
		return nil, fmt.Errorf(errorClientSend, err)
	}
	return byteResponse, nil
}

func (c HTTPClient) parseResponse(data []byte) error {
	responseParsed := responseRoot{}
	err := json.Unmarshal(data, &responseParsed)
	if err != nil {
		return fmt.Errorf("got error while unmarshal data (%s): %v", string(data), err)
	}

	if responseParsed.Result.Status != "OK" {
		return fmt.Errorf("sender answered with error status: %s", string(data))
	}
	return nil
}

func (c HTTPClient) getBaseURL(campaignSlug string) string {
	return fmt.Sprintf(
		"%s/api/0/%s/transactional/%s/",
		c.config.URL,
		c.config.Account,
		campaignSlug,
	)
}

func (c HTTPClient) getData(toEmail string) url.Values {
	return url.Values{
		"async":       {"true"},
		"to_email":    {toEmail},
		"for_testing": {c.config.Testing},
	}
}

func (c *HTTPClient) retryRequest(request func() error) error {
	var err error
	requestBackoff := &backoff.ExponentialBackOff{
		InitialInterval:     backoff.DefaultInitialInterval,
		RandomizationFactor: backoff.DefaultRandomizationFactor,
		Multiplier:          backoff.DefaultMultiplier,
		MaxInterval:         100 * time.Millisecond,
		MaxElapsedTime:      1 * time.Second,
		Clock:               backoff.SystemClock,
		Stop:                backoff.Stop,
	}
	requestBackoff.Reset()

	err = backoff.Retry(request, requestBackoff)
	if err != nil {
		return fmt.Errorf(errorClientDoRequest, err)
	}
	return nil
}
