package startrek

import (
	"bytes"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/core/log"
)

const (
	TokenHeader = "Authorization"
	OrgIDHeader = "X-Org-Id"
)

type Client interface {
	PostIssue(body []byte) ([]byte, error)
	SearchIssues(body []byte) ([]byte, error)
	PatchIssue(issueKey string, body []byte) ([]byte, error)
}

type HTTPClient struct {
	backendBaseURL string
	token          string
	orgID          string
	httpClient     *http.Client
	logger         log.Logger
}

func NewHTTPClient(backendBaseURL, token, orgID string, httpClient *http.Client, logger log.Logger) *HTTPClient {
	return &HTTPClient{
		backendBaseURL: backendBaseURL,
		token:          token,
		orgID:          orgID,
		httpClient:     httpClient,
		logger:         logger,
	}
}

func (c HTTPClient) PostIssue(body []byte) ([]byte, error) {
	result, err := c.issueRequest("POST", "/issues/", body)
	if err != nil {
		return nil, errorPostIssue.Wrap(err)
	}
	return result, nil
}

func (c HTTPClient) SearchIssues(body []byte) ([]byte, error) {
	result, err := c.issueRequest("POST", "/issues/_search/", body)
	if err != nil {
		return nil, errorSearchIssues.Wrap(err)
	}
	return result, nil
}

func (c HTTPClient) PatchIssue(issueKey string, body []byte) ([]byte, error) {
	result, err := c.issueRequest("PATCH", "/issues/"+issueKey, body)
	if err != nil {
		return nil, errorSearchIssues.Wrap(err)
	}
	return result, nil
}

func (c *HTTPClient) issueRequest(method, location string, body []byte) ([]byte, error) {
	var byteResponse []byte
	var statusCode int

	request := func() error {
		var bodyReader io.Reader
		if body != nil {
			bodyReader = bytes.NewReader(body)
		}
		url := c.backendBaseURL + location
		req, err := http.NewRequest(method, url, bodyReader)
		if err != nil {
			return errorIssueRequest.Wrap(err)
		}
		c.injectAuth(req)

		response, err := c.httpClient.Do(req)
		if err != nil {
			return errorIssueRequest.Wrap(err)
		}
		defer response.Body.Close()

		byteResponse, err = ioutil.ReadAll(response.Body)
		statusCode = response.StatusCode
		if response.StatusCode >= 400 && response.StatusCode != http.StatusNotFound {
			return fmt.Errorf("startrek answer with code %d: %s", response.StatusCode, string(byteResponse))
		}

		if err != nil {
			return errorIssueRequest.Wrap(err)
		}
		return nil
	}
	err := c.retryRequest(request)
	if err != nil {
		return nil, errorIssueRequest.Wrap(err)
	}
	if statusCode == http.StatusNotFound {
		return nil, ErrorNotFound
	}
	return byteResponse, nil
}

func (c HTTPClient) injectAuth(r *http.Request) {
	r.Header.Set(TokenHeader, "OAuth "+c.token)
	r.Header.Set(OrgIDHeader, c.orgID)
}

func (c *HTTPClient) retryRequest(request func() error) error {
	var err error
	requestBackoff := &backoff.ExponentialBackOff{
		InitialInterval:     backoff.DefaultInitialInterval,
		RandomizationFactor: backoff.DefaultRandomizationFactor,
		Multiplier:          backoff.DefaultMultiplier,
		MaxInterval:         100 * time.Millisecond,
		MaxElapsedTime:      1 * time.Second,
		Clock:               backoff.SystemClock,
		Stop:                backoff.Stop,
	}
	requestBackoff.Reset()

	err = backoff.Retry(request, requestBackoff)
	if err != nil {
		return errorDoRequest.Wrap(err)
	}
	return nil
}
