package startrek

import (
	"encoding/json"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/model"
)

type Controller struct {
	logger log.Logger
	client Client
	config *configs.Startrek
}

func NewController(
	logger log.Logger,
	config *configs.Startrek,
) *Controller {
	httpClient := &http.Client{
		Timeout: time.Duration(config.Timeout) * time.Millisecond,
	}
	client := NewHTTPClient(
		config.BaseURL,
		config.Token,
		config.OrgID,
		httpClient, logger,
	)

	return &Controller{
		logger: logger,
		client: client,
		config: config,
	}
}

type searchRequest struct {
	Filter filterSearch `json:"filter"`
}

type filterSearch struct {
	Queue  string `json:"queue"`
	Tags   string `json:"tags,omitempty"`
	Unique string `json:"unique,omitempty"`
}

func (c Controller) search(request searchRequest) (result []Issue, err error) {
	body, err := json.Marshal(request)
	if err != nil {
		return nil, errorControllerGetMailingIssues.Wrap(err)
	}

	body, err = c.client.SearchIssues(body)
	if err != nil {
		return nil, errorControllerGetMailingIssues.Wrap(err)
	}

	err = json.Unmarshal(body, &result)
	if err != nil {
		return nil, errorControllerGetMailingIssues.Wrap(err)
	}

	return result, nil
}

func (c Controller) GetIssue(issueID string) (Issue, error) {
	query := searchRequest{
		Filter: filterSearch{
			Queue:  c.config.Queue,
			Unique: issueID,
		},
	}
	result, err := c.search(query)
	if err != nil {
		return Issue{}, errorControllerGetIssue.Wrap(err)
	}
	if len(result) == 0 {
		return Issue{}, ErrorNotFound
	}
	if len(result) > 1 {
		return Issue{}, ErrorAmbiguous
	}

	return result[0], nil
}

func (c Controller) GetMailingIssues() (result []Issue, err error) {
	query := searchRequest{
		Filter: filterSearch{
			Queue: c.config.Queue,
			Tags:  "mailing",
		},
	}
	return c.search(query)
}

func (c Controller) PostIssue(story model.Story) error {
	description, err := BuildDescription(story)
	if err != nil {
		return errorControllerPostIssue.Wrap(err)
	}

	storyRequest := Issue{
		Summary:     story.Title,
		Queue:       IssueQueue{ID: c.config.QueueNo},
		Description: description,
		Unique:      story.UID,
		Tags:        []string{"mailing"},
	}
	body, err := json.Marshal(storyRequest)
	if err != nil {
		return errorControllerPostIssue.Wrap(err)
	}

	_, err = c.client.PostIssue(body)
	if err != nil {
		return errorControllerPostIssue.Wrap(err)
	}
	return nil
}

type removeMailingTagRequest struct {
	Tags removeMailingTag `json:"tags"`
}

type removeMailingTag struct {
	Remove []string `json:"remove,omitempty"`
}

func (c Controller) RemoveMailingTag(issue Issue) error {
	query := removeMailingTagRequest{
		Tags: removeMailingTag{
			Remove: []string{"mailing"},
		},
	}

	body, err := json.Marshal(query)
	if err != nil {
		return errorControllerRemoveMailingTag.Wrap(err)
	}

	_, err = c.client.PatchIssue(*issue.Key, body)
	if err != nil {
		return errorControllerRemoveMailingTag.Wrap(err)
	}

	return nil
}
