package startrek

import (
	"bytes"
	"strings"
	"text/template"

	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/model"
)

const (
	descriptionTpl = `
Код истории:
{{.UID}}

Название рассказа:
{{.Title}}

Тип рассказа:
{{.Type}}

{{.Blocks}}

Нажимая,  я соглашаюсь на участие в конкурсе и на подписку:
Да
`
	videoTpl = `
Блок {{.No}}:
<[{{with .Text -}}
Комментарий к видео:
{{.}}{{end}}

{{with .URLs}}Видео:
{{range $url := .}}  {{$url}}
{{end}}{{end}}
]>
`
	photoTpl = `
Блок {{.No}}:
<[{{with .Text -}}
Комментарий к фото:
{{.}}{{end}}

{{with .URLs}}Фото:
{{range $url := .}}  {{$url}}
{{end}}{{end}}
]>
`
)

var (
	preparedDescTpl  = template.Must(template.New("issue").Parse(descriptionTpl))
	preparedVideoTpl = template.Must(template.New("video").Parse(videoTpl))
	preparedPhotoTpl = template.Must(template.New("photo").Parse(photoTpl))
)

type descriptionParams struct {
	UID    string
	Title  string
	Type   string
	Blocks string
}

type blockParams struct {
	No   int
	Text *string
	URLs []string
}

func BuildDescription(story model.Story) (string, error) {
	blockData, err := buildBlocks(story)
	if err != nil {
		return "", errorBuildDesc.Wrap(err)
	}

	desc := descriptionParams{
		UID:    story.UID,
		Title:  story.Title,
		Type:   getStoryType(story),
		Blocks: blockData,
	}

	var buf bytes.Buffer
	err = preparedDescTpl.Execute(&buf, desc)
	if err != nil {
		return "", errorBuildDesc.Wrap(err)
	}
	return buf.String(), nil
}

func getStoryType(story model.Story) string {
	switch story.Type {
	case model.Video:
		return "Видео"
	case model.TextAndVideo:
		return "Видеорассказ"
	case model.TextAndPhoto:
		return "Фоторассказ"
	default:
		return "unexpected"
	}
}

func buildBlocks(story model.Story) (string, error) {
	var tpl *template.Template
	var data []string

	if story.Type == model.TextAndVideo || story.Type == model.Video {
		tpl = preparedVideoTpl
	} else {
		tpl = preparedPhotoTpl
	}

	for blockNo, block := range story.Blocks {
		buf := bytes.Buffer{}
		urls := make([]string, 0)
		for _, upload := range block.Uploads {
			urls = append(urls, upload.PublicURL)
		}

		err := tpl.Execute(
			&buf,
			blockParams{
				No:   blockNo + 1,
				Text: block.Text,
				URLs: urls,
			},
		)
		if err != nil {
			return "", errorBuildBlocks.Wrap(err)
		}
		data = append(data, buf.String())
	}

	return strings.Join(data, "\n"), nil
}
