package handler

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/travel/library/go/httputil"
)

type newStoryRequest struct {
	YandexUID *string `json:"yandexUid"`
}

type newStoryResponse struct {
	StoryID string `json:"storyId"`
}

func (h *HTTPStoriesHandler) NewStory(w http.ResponseWriter, r *http.Request) {
	ctx, ctxCancel := context.WithTimeout(r.Context(), 2*time.Second)
	defer ctxCancel()

	span, ctx := opentracing.StartSpanFromContext(ctx, "internal.handler.HTTPStoriesHandler: NewStory")
	defer span.Finish()

	var err error
	defer func() {
		if err != nil {
			h.logger.Errorf("internal.story.handler.NewStory: %v", err)
		}
	}()

	query := new(newStoryRequest)
	err = json.NewDecoder(r.Body).Decode(query)
	if err != nil {
		httputil.HandleError(
			fmt.Errorf("an error occured while parsing json: %s", err),
			http.StatusBadRequest, w,
		)
		return
	}

	if query.YandexUID == nil {
		httputil.HandleError(fmt.Errorf("missing required param 'yandexUid'"), http.StatusBadRequest, w)
		return
	}

	tx, err := h.pool.Begin(ctx)
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		}
	}()

	story, err := h.repository.NewStory(ctx, h.pool, *query.YandexUID)
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}

	if err = tx.Commit(ctx); err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}

	resultBytes, err := json.Marshal(newStoryResponse{StoryID: story.UID})
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}

	w.WriteHeader(http.StatusOK)
	w.Header().Set("Content-Type", "application/json")
	_, _ = w.Write(resultBytes)
}
