package handler

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/travel/library/go/httputil"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/story/httpspec"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/story/mapping"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/story/validate"
)

func (h *HTTPStoriesHandler) SendStory(w http.ResponseWriter, r *http.Request) {
	ctx, ctxCancel := context.WithTimeout(r.Context(), 2*time.Second)
	defer ctxCancel()

	span, ctx := opentracing.StartSpanFromContext(ctx, "internal.handler.HTTPStoriesHandler: SendStory")
	defer span.Finish()

	var err error
	defer func() {
		if err != nil {
			h.logger.Errorf("story.handler.SendStory: %v", err)
		}
	}()

	query := new(httpspec.SendStoryRequest)
	err = json.NewDecoder(r.Body).Decode(query)
	if err != nil {
		httputil.HandleError(
			fmt.Errorf("an error occured while parsing json: %s", err),
			http.StatusBadRequest, w,
		)
		return
	}

	err = validate.SendStoryRequest(query)
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}

	story, err := mapping.MapSendStoryRequest(query, h.config)
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}

	tx, err := h.pool.Begin(ctx)
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		}
	}()

	err = h.repository.UpdateStory(ctx, tx, story)
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}

	if err = tx.Commit(ctx); err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}

	resultBytes, err := json.Marshal(httpspec.SendStoryResponse{})
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	w.WriteHeader(http.StatusOK)
	w.Header().Set("Content-Type", "application/json")
	_, _ = w.Write(resultBytes)
}
