package mapping

import (
	"fmt"

	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/model"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/service/s3"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/story/httpspec"
)

func MapSendStoryRequest(request *httpspec.SendStoryRequest, config *configs.Config) (*model.Story, error) {
	storyType, err := mapStoryType(request.StoryType)
	if err != nil {
		return nil, err
	}

	storyBlocks := make([]model.StoryBlock, 0, len(request.Blocks))

	for i, block := range request.Blocks {
		storyBlock, err := mapStoryBlock(request.StoryID, block, config)
		if err != nil {
			return nil, fmt.Errorf("Blocks.%d.Upload: %w", i, err)
		}
		storyBlocks = append(storyBlocks, *storyBlock)
	}
	story := new(model.Story)
	story.Blocks = storyBlocks

	story.UID = request.StoryID
	story.Email = request.Email
	story.FirstName = request.FirstName
	story.LastName = request.LastName
	story.Title = request.Title
	story.Type = storyType

	return story, nil
}

func mapStoryBlock(storyUID string, block httpspec.StoryBlock, config *configs.Config) (*model.StoryBlock, error) {
	if block.Text == nil && len(block.UploadIDs) == 0 {
		return nil, fmt.Errorf("block should contain text or uploads")
	}

	storyBlock := &model.StoryBlock{StoryUID: storyUID}

	if block.Text != nil {
		storyBlock.Text = block.Text
	}

	uploadCount := len(block.UploadIDs)
	if uploadCount != 0 {
		storyBlock.Uploads = make([]model.StoryBlockUpload, uploadCount)
		for i, uploadID := range block.UploadIDs {
			storyBlock.Uploads[i] = model.StoryBlockUpload{
				UploadUID: uploadID,
				PublicURL: s3.MakeURL(uploadID, &config.S3),
			}
		}
	}
	return storyBlock, nil
}

func mapStoryType(source string) (model.StoryType, error) {
	switch source {
	case "video":
		return model.Video, nil
	case "text_and_video":
		return model.TextAndVideo, nil
	case "text_and_photo":
		return model.TextAndPhoto, nil
	default:
		return 0, fmt.Errorf("storyType: unexpected story type")
	}
}
