package validate

import (
	"fmt"
	"regexp"
	"strings"

	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/story/httpspec"
)

var rxUUID4 = regexp.MustCompile("^[0-9a-f]{8}-[0-9a-f]{4}-4[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$")

func SendStoryRequest(request *httpspec.SendStoryRequest) error {
	var errors []string
	if !rxUUID4.MatchString(request.StoryID) {
		message := fmt.Sprintf("storyId: %s does not validate as uuidv4", request.StoryID)
		errors = append(errors, message)
	}
	if len(request.Email) == 0 {
		errors = append(errors, "email: haven't to be empty")
	}
	if len(request.FirstName) == 0 {
		errors = append(errors, "firstName: haven't to be empty")
	}
	if len(request.LastName) == 0 {
		errors = append(errors, "lastName: haven't to be empty")
	}
	if len(request.Title) == 0 {
		errors = append(errors, "title: haven't to be empty")
	}
	for i, block := range request.Blocks {
		for j, uploadID := range block.UploadIDs {
			if !rxUUID4.MatchString(uploadID) {
				message := fmt.Sprintf(
					"blocks.%d.uploadIds[%d]: %s does not validate as uuidv4",
					i, j, uploadID,
				)
				errors = append(errors, message)
			}
		}
	}
	if len(errors) > 0 {
		return fmt.Errorf("validation error: %s", strings.Join(errors, "; "))
	}
	return nil
}
