package tracker

import (
	"context"
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"github.com/jackc/pgx/v4/pgxpool"

	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/metrics"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/model"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/repository"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/service/startrek"
)

var (
	modulePrefix = "tracker"

	errorRetrievingStories = xerrors.NewSentinel(fmt.Sprintf("%s.Controller.getStoriesWithoutIssues", modulePrefix))
	errorSaveStory         = xerrors.NewSentinel(fmt.Sprintf("%s.Controller.updateStory", modulePrefix))
	errorPostIssue         = xerrors.NewSentinel(fmt.Sprintf("%s.Controller.createIssue", modulePrefix))
)

type Controller struct {
	logger             log.Logger
	pool               *pgxpool.Pool
	s3conf             *configs.S3
	startrekController *startrek.Controller
	storyRepository    *repository.StoryRepository
}

func NewController(
	logger log.Logger,
	pool *pgxpool.Pool,
	s3conf *configs.S3,
	startrekController *startrek.Controller,
	storyRepository *repository.StoryRepository,
) *Controller {
	return &Controller{
		logger:             logger,
		pool:               pool,
		s3conf:             s3conf,
		startrekController: startrekController,
		storyRepository:    storyRepository,
	}
}

func (c *Controller) getStoriesWithoutIssues(ctx context.Context) ([]*model.Story, error) {
	tx, err := c.pool.Begin(ctx)
	if err != nil {
		return nil, errorRetrievingStories.Wrap(err)
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		} else {
			_ = tx.Commit(ctx)
		}
	}()
	return c.storyRepository.GetStoriesWithoutIssues(ctx, tx)
}

func (c *Controller) createIssue(ctx context.Context, story *model.Story) error {
	tx, err := c.pool.Begin(ctx)
	if err != nil {
		return errorPostIssue.Wrap(err)
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		}
	}()

	err = c.storyRepository.PopulateStoryWithBlocks(ctx, tx, story, c.s3conf)

	if err != nil {
		return errorPostIssue.Wrap(err)
	}

	err = c.startrekController.PostIssue(*story)
	if err != nil {
		return errorPostIssue.Wrap(err)
	}
	if err := tx.Commit(ctx); err != nil {
		return errorSaveStory.Wrap(err)
	}

	return c.updateStory(ctx, story)
}

func (c *Controller) updateStory(ctx context.Context, story *model.Story) error {
	tx, err := c.pool.Begin(ctx)
	if err != nil {
		return errorSaveStory.Wrap(err)
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		}
	}()

	err = c.storyRepository.UpdateStoryState(ctx, tx, story, model.TicketCreated)
	if err != nil {
		return errorSaveStory.Wrap(err)
	}
	if err := tx.Commit(ctx); err != nil {
		return errorSaveStory.Wrap(err)
	}

	return nil
}

func (c *Controller) runOnce(ctx context.Context) {
	c.logger.Info("Searching for stories without issues")

	stories, err := c.getStoriesWithoutIssues(ctx)
	metrics.TrackerRegistry.SetStoriesWithoutIssues(len(stories))
	if err != nil {
		c.logger.Structured().Error("tracker syncing fail", log.Error(err))
		return
	}

	for _, s := range stories {
		_, err := c.startrekController.GetIssue(s.UID)
		if err == nil {
			// we already have such issue, then should update it in db
			err = c.updateStory(ctx, s)
			if err != nil {
				c.logger.Structured().Error("failed to update issue", log.Error(err))
				metrics.TrackerRegistry.AddError()
			}
			continue
		}
		if errors.Is(err, startrek.ErrorNotFound) {
			err = c.createIssue(ctx, s)
			if err != nil {
				c.logger.Structured().Error("failed to create issue", log.Error(err))
				metrics.TrackerRegistry.AddError()
			}

		} else {
			c.logger.Structured().Error("request to tracker failed", log.Error(err))
			metrics.TrackerRegistry.AddError()
		}
	}
}

func (c *Controller) Run(ctx context.Context) {
	for {
		select {
		case <-time.After(time.Minute):
			c.runOnce(ctx)
		case <-ctx.Done():
			c.logger.Structured().Info("finish tracker cycle")
			return
		}
	}
}
