package handler

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"os"
	"path"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/jackc/pgx/v4/pgxpool"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/httputil"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/repository"
)

type HTTPHooksHandler struct {
	config           *configs.Config
	pool             *pgxpool.Pool
	storyRepository  *repository.StoryRepository
	uploadRepository *repository.UploadRepository
	logger           log.Logger
}

func NewHTTPHooksHandler(
	config *configs.Config,
	pool *pgxpool.Pool,
	storyRepository *repository.StoryRepository,
	uploadRepository *repository.UploadRepository,
	l log.Logger,
) *HTTPHooksHandler {
	return &HTTPHooksHandler{
		config:           config,
		pool:             pool,
		storyRepository:  storyRepository,
		uploadRepository: uploadRepository,
		logger:           l,
	}
}

func (h *HTTPHooksHandler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Post("/tusd-hook", h.HandleHook)
	}
}

type metadata struct {
	StoryID  string `json:"storyId"`
	UploadID string `json:"uploadId"`
	FileName string `json:"filename"`
	FileType string `json:"filetype"`
}

type storage struct {
	Path string `json:"Path"`
}

type upload struct {
	ID       string   `json:"ID"`
	Metadata metadata `json:"Metadata"`
	Storage  storage  `json:"Storage"`
}

type hookRequest struct {
	Upload upload `json:"Upload"`
}

func (h *HTTPHooksHandler) HandleHook(w http.ResponseWriter, r *http.Request) {
	ctx, ctxCancel := context.WithTimeout(r.Context(), time.Second)
	defer ctxCancel()

	span, ctx := opentracing.StartSpanFromContext(ctx, "upload.handler.HTTPHooksHandler: HandleHook")
	defer span.Finish()

	var err error
	defer func() {
		if err != nil {
			h.logger.Errorf("upload.handler.HandleHook: %v", err)
		}
	}()

	hookNames, found := r.Header["Hook-Name"]
	if !found || len(hookNames) == 0 {
		w.WriteHeader(http.StatusOK)
		return
	}

	h.logger.Info("got_request", log.String("hookName", hookNames[0]))

	request := new(hookRequest)
	err = json.NewDecoder(r.Body).Decode(request)
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}

	if hookNames[0] == "pre-create" {
		h.checkExistenceOfStory(ctx, request, w)
		return
	}

	if hookNames[0] == "post-create" {
		h.createUpload(ctx, request, w)
	}

	if hookNames[0] == "post-finish" {
		h.moveUploadToReady(request, w)
	}

	w.WriteHeader(http.StatusOK)
}

func (h *HTTPHooksHandler) checkExistenceOfStory(ctx context.Context, request *hookRequest, w http.ResponseWriter) {
	storyID := request.Upload.Metadata.StoryID
	story, err := h.storyRepository.GetStory(ctx, h.pool, storyID)
	if err != nil {
		h.logger.Errorf("getting_story: %v", err)
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	if story == nil {
		err = fmt.Errorf("a story was not found (storyId=%s)", storyID)
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	w.WriteHeader(http.StatusOK)
}

func (h *HTTPHooksHandler) createUpload(ctx context.Context, request *hookRequest, w http.ResponseWriter) {
	requestMetadata := request.Upload.Metadata

	tx, err := h.pool.Begin(ctx)
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		}
	}()

	_, err = h.uploadRepository.NewUpload(
		ctx, tx,
		requestMetadata.UploadID,
		request.Upload.ID,
		requestMetadata.StoryID,
		requestMetadata.FileName,
		requestMetadata.FileType,
	)
	if err != nil {
		h.logger.Errorf("creating_upload: %v", err)
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}

	if err = tx.Commit(ctx); err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}

	w.WriteHeader(http.StatusOK)
}

func (h *HTTPHooksHandler) moveUploadToReady(request *hookRequest, w http.ResponseWriter) {
	sourcePath := request.Upload.Storage.Path
	if sourcePath == "" {
		w.WriteHeader(http.StatusOK)
		return
	}

	_, file := path.Split(sourcePath)
	destinationPath := path.Join(h.config.S3.ReadyFileDir, file)

	err := os.Rename(sourcePath, destinationPath)
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}

	err = os.Rename(sourcePath+".info", destinationPath+".info")
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	h.logger.Infof("move files: %s -> %s", sourcePath, destinationPath)

	w.WriteHeader(http.StatusOK)
}
