package uploader

import (
	"context"
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"strings"
	"time"

	"github.com/jackc/pgx/v4/pgxpool"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/app/configs"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/metrics"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/model"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/repository"
	"a.yandex-team.ru/travel/marketing/folk_guide_contest/internal/pkg/service/s3"
)

var (
	modulePrefix = "uploader"

	errorUploadFile = xerrors.NewSentinel(fmt.Sprintf("%s.uploadFile", modulePrefix))
	errorRemoveFile = xerrors.NewSentinel(fmt.Sprintf("%s.removeFile", modulePrefix))
)

type Uploader struct {
	logger           log.Logger
	config           *configs.S3
	pool             *pgxpool.Pool
	uploadRepository *repository.UploadRepository
	client           *s3.Client
}

func NewUploader(
	config *configs.S3,
	client *s3.Client,
	uploadRepository *repository.UploadRepository,
	pool *pgxpool.Pool,
	logger log.Logger,
) *Uploader {
	return &Uploader{
		logger:           logger,
		config:           config,
		pool:             pool,
		uploadRepository: uploadRepository,
		client:           client,
	}
}

func (u Uploader) RunUploading(ctx context.Context) {
	var err error
	for {
		select {
		case <-time.After(time.Minute):
			err = u.UploadFiles(ctx)
			if err != nil {
				u.logger.Errorf("RunUploading: %v", err)
				return
			}
			err = u.RemoveFiles(ctx)
			if err != nil {
				u.logger.Errorf("RunUploading: %v", err)
				return
			}
		case <-ctx.Done():
			u.logger.Infof("finish uploading cycle")
			return
		}
	}
}

func (u Uploader) UploadFiles(ctx context.Context) error {
	files, err := ioutil.ReadDir(u.config.ReadyFileDir)
	if err != nil {
		return fmt.Errorf("uploader.UploadFiles: %w", err)
	}

	u.logger.Infof("uploading files (dir=%s)", u.config.ReadyFileDir)

	metrics.UploaderRegistry.SetQueueLength(len(files))
	for _, f := range files {
		if ctx.Err() != nil {
			return nil
		}

		name := f.Name()
		if strings.HasSuffix(name, ".info") {
			continue
		}
		err := u.uploadFile(ctx, name)
		if err != nil {
			metrics.UploaderRegistry.AddError()
			u.logger.Errorf("upload_error (tusID=%s): %v", name, err)
			continue
		}
	}

	return nil
}

func (u Uploader) uploadFile(ctx context.Context, tusID string) error {
	u.logger.Infof("uploading file (tusID=%s)", tusID)

	upload, err := u.uploadRepository.GetUploadByTusID(ctx, u.pool, tusID)
	if err != nil {
		return errorUploadFile.Wrap(err)
	}
	if upload == nil {
		return nil
	}
	if upload.Status != model.UploadInit {
		return nil
	}

	tx, err := u.pool.Begin(ctx)
	if err != nil {
		return errorUploadFile.Wrap(err)
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		}
	}()

	upload.Status = model.UploadUploaded
	err = u.uploadRepository.UpdateUpload(ctx, tx, upload)
	if err != nil {
		return errorUploadFile.Wrap(err)
	}

	file, err := os.Open(path.Join(u.config.ReadyFileDir, tusID))
	if err != nil {
		return errorUploadFile.Wrap(err)
	}

	err = u.client.PutObject(ctx, upload.UID, file, upload.FileType)
	if err != nil {
		return errorUploadFile.Wrap(err)
	}

	if err = tx.Commit(ctx); err != nil {
		return errorUploadFile.Wrap(err)
	}
	return nil
}

func (u Uploader) RemoveFiles(ctx context.Context) error {
	files, err := ioutil.ReadDir(u.config.ReadyFileDir)
	if err != nil {
		return fmt.Errorf("uploader.RemoveFiles: %w", err)
	}

	u.logger.Infof("removing files (dir=%s)", u.config.ReadyFileDir)
	for _, f := range files {
		if ctx.Err() != nil {
			return nil
		}

		name := f.Name()
		if strings.HasSuffix(name, ".info") {
			continue
		}
		err := u.removeFile(ctx, name)
		if err != nil {
			metrics.UploaderRegistry.AddError()
			u.logger.Errorf("remove_error (tusID=%s): %v", name, err)
			continue
		}
	}

	return nil
}

func (u Uploader) removeFile(ctx context.Context, tusID string) error {
	u.logger.Infof("removing file (tusID=%s)", tusID)

	upload, err := u.uploadRepository.GetUploadByTusID(ctx, u.pool, tusID)
	if err != nil {
		return errorRemoveFile.Wrap(err)
	}
	if upload == nil {
		return nil
	}
	if upload.Status != model.UploadUploaded {
		return nil
	}

	tx, err := u.pool.Begin(ctx)
	if err != nil {
		return errorRemoveFile.Wrap(err)
	}
	defer func() {
		if err != nil {
			_ = tx.Rollback(ctx)
		}
	}()

	upload.Status = model.UploadFileRemoved
	err = u.uploadRepository.UpdateUpload(ctx, tx, upload)
	if err != nil {
		return errorRemoveFile.Wrap(err)
	}

	err = os.Remove(path.Join(u.config.ReadyFileDir, tusID))
	if err != nil {
		return errorRemoveFile.Wrap(err)
	}

	_ = os.Remove(path.Join(u.config.ReadyFileDir, tusID+".info"))

	if err = tx.Commit(ctx); err != nil {
		return errorRemoveFile.Wrap(err)
	}
	return nil
}
