from dataclasses import dataclass
from enum import Enum
from time import sleep

from requests import Session
from requests.adapters import HTTPAdapter
from requests.exceptions import HTTPError
from requests.packages.urllib3.util.retry import Retry

from tvmauth import TvmClient, TvmClientStatus, TvmApiClientSettings


@dataclass
class TopupRequest:
    transaction_id: str
    passport_id: str
    campaign_id: str
    amount: int
    payload: dict[str, str]


class RequestStatus(Enum):
    SCHEDULED = 'SCHEDULED'
    VALIDATION_ERROR = 'VALIDATION_ERROR'


class TopupStatus(Enum):
    NOT_STARTED = 'NOT_STARTED'
    RUNNING = 'RUNNING'
    COMPLETED = 'COMPLETED'
    FAILED = 'FAILED'


class MediaBillingClient:

    __rps_max__ = 10

    def __init__(self, url: str, tvm_service_id: str, tvm_client_id: str, tvm_secret: str):
        self.url = url
        tvm_client = self._create_tvm_client(tvm_service_id, tvm_client_id, tvm_secret)
        tvm_ticket = tvm_client.get_service_ticket_for('travel_marketing_client')
        self.session = self._get_session(tvm_ticket)

    def request_topup(self, request: TopupRequest) -> None:
        sleep(1 / self.__rps_max__)

        url = f'{self.url}/points-topup'
        req = {
            'transactionId': request.transaction_id,
            'uid': request.passport_id,
            'pointsCampaignId': request.campaign_id,
            'pointsAmount': request.amount,
            'payload': request.payload,
        }
        rsp = self.session.post(url, json=req)
        rsp.raise_for_status()
        result = rsp.json()['result']
        status = RequestStatus(result['status'])
        if status != RequestStatus.SCHEDULED:
            error_message = result.get('error')
            raise Exception(f'Failed to request topup: {status, error_message}')

    def get_topup_status(self, transaction_id: str) -> TopupStatus:
        sleep(1 / self.__rps_max__)

        url = f'{self.url}/points-topup/{transaction_id}'
        rsp = self.session.get(url)
        if 500 <= rsp.status_code < 600:
            http_error_msg = f'{rsp.status_code} Server Error: {rsp.reason} for url: {rsp.url}'
            raise HTTPError(http_error_msg, response=rsp)
        if rsp.status_code == 404:
            return TopupStatus.NOT_STARTED
        return TopupStatus(rsp.json()['result']['status'])

    @staticmethod
    def _create_tvm_client(tvm_service_id, tvm_client_id, tvm_secret):
        destinations = {'travel_marketing_client': tvm_service_id}
        settings = TvmApiClientSettings(
            self_tvm_id=tvm_client_id,
            enable_service_ticket_checking=True,
            enable_user_ticket_checking=False,
            self_secret=tvm_secret,
            dsts=destinations,
        )
        client = TvmClient(settings)
        if client.status != TvmClientStatus.Ok:
            raise RuntimeError(f'Bad tvm client status: {client.status}')
        return client

    @staticmethod
    def _get_session(tvm_ticket: str):
        session = Session()
        retry = Retry(
            total=3,
            backoff_factor=2.0,
            status_forcelist=frozenset([500, 503, 413, 429])
        )
        session.mount('http://', HTTPAdapter(max_retries=retry))
        session.mount('https://', HTTPAdapter(max_retries=retry))
        session.headers.update({'X-Ya-Service-Ticket': tvm_ticket})
        return session
