import logging
from dataclasses import dataclass
from datetime import datetime, timezone, timedelta, date
from typing import Any

from email_validator import validate_email, EmailNotValidError

from travel.marketing.tools.lototron.event import Event
from travel.marketing.tools.lototron.travel_api_client import OrderType


HOTEL_ORDER_TYPES = {
    OrderType.HOTEL_BNOVO,
    OrderType.HOTEL_DOLPHIN,
    OrderType.HOTEL_EXPEDIA,
    OrderType.HOTEL_TRAVELLINE,
}


RawOrder = dict[str, Any]


@dataclass
class Order:
    order_type: OrderType
    partner_order_id: str
    created_at: datetime
    order_status: str
    email: str
    email_hidden: str
    phone: str
    phone_hidden: str
    passport_id: str
    login: str
    order_amount: float
    points_to_topup: int
    withdraw_points: int

    @classmethod
    def from_raw_order(cls, raw_order: RawOrder, tz: timezone) -> 'Order':
        order_type = raw_order['order_type']
        personal_user_data = raw_order['personal_user_data']
        email = cls.get_normalized_email(personal_user_data['email'])
        email_hidden = cls._get_email_hidden(email)
        phone = cls._get_normalized_phone(personal_user_data['phone'])
        phone_hidden = cls._get_phone_hidden(phone)
        passport_id = personal_user_data['owner_passport_id']
        login = personal_user_data['owner_login']
        order_amount = float(raw_order['amount']['value'])
        topup_points = 0
        withdraw_points = 0
        yandex_plus = raw_order.get('yandex_plus')
        if yandex_plus is not None:
            topup_points = yandex_plus.get('topup_points') or 0
            withdraw_points = yandex_plus.get('withdraw_points') or 0
        return Order(
            order_type=order_type,
            partner_order_id=raw_order['partner_order_id'],
            created_at=parse_datetime_tz(raw_order['created_at'], tz),
            order_status=raw_order['order_status'],
            email=email,
            email_hidden=email_hidden,
            phone=phone,
            phone_hidden=phone_hidden,
            passport_id=passport_id,
            login=login,
            order_amount=order_amount,
            points_to_topup=int(order_amount - topup_points),
            withdraw_points=withdraw_points,
        )

    @staticmethod
    def get_normalized_email(email: str) -> str:
        try:
            email = validate_email(email, check_deliverability=False).ascii_email
        except EmailNotValidError as e:
            logging.warning(e)
            email = ''
        return email

    @staticmethod
    def _get_email_hidden(email: str) -> str:
        if not email:
            return email
        email = validate_email(email, check_deliverability=False)
        local_part = email.ascii_local_part
        visible_count = 3 if len(local_part) >= 6 else 1
        return f'{local_part[:visible_count]}***@{email.ascii_domain}'

    @staticmethod
    def _get_normalized_phone(phone: str) -> str:
        normalized_chars = list()
        for char in phone:
            if char == '+' and not normalized_chars:
                normalized_chars.append(char)
                continue
            if char.isdecimal():
                normalized_chars.append(char)
        normalized_phone = ''.join(normalized_chars)
        normalized_phone = '*' * (11 - len(normalized_phone)) + normalized_phone
        return normalized_phone

    @staticmethod
    def _get_phone_hidden(phone: str) -> str:
        country_code = phone[0:-10]
        first_pair = phone[-4:-2]
        second_pair = phone[-2:]
        return f'{country_code}(***) ***-{first_pair}-{second_pair}'


@dataclass
class CpaOrder:
    partner_order_id: str
    passport_id: str
    order_status: str
    order_amount: float
    profit_amount: float
    topup_points: int
    points_to_topup: int
    topup_time: datetime

    @classmethod
    def from_raw_order(cls, raw_order: RawOrder, tz: timezone) -> 'CpaOrder':
        try:
            return CpaOrder(
                partner_order_id=raw_order['partner_order_id'],
                passport_id='',
                order_status=raw_order['status'],
                order_amount=raw_order['order_amount'],
                profit_amount=raw_order['profit_amount'],
                topup_points=raw_order.get('yandex_plus_topup_points') or 0,
                points_to_topup=0,
                topup_time=cls._get_topup_time(raw_order, tz),
            )
        except Exception as e:
            raise Exception(f'{raw_order}, {e}')

    @staticmethod
    def _get_topup_time(raw_order: RawOrder, tz: timezone):
        category = raw_order['category']
        if category == 'avia':
            finish_date = raw_order['date_backward'] or raw_order['date_forward']
            finish_time = parse_datetime_tz(finish_date, tz)
        elif category == 'hotels':
            finish_time = Event.date_to_datetime(date.fromisoformat(raw_order['check_out']), tz)
        elif category == 'train':
            finish_time = datetime.fromtimestamp(raw_order['arrival'], tz)
        else:
            raise RuntimeError(f'Unsupported category: {category}')
        return finish_time + timedelta(days=1)


def parse_datetime_tz(s: str, tz: timezone) -> datetime:
    return datetime.fromisoformat(s).replace(tzinfo=timezone.utc).astimezone(tz)
