from requests import Session
from requests.adapters import HTTPAdapter
from requests.auth import HTTPBasicAuth
from requests.packages.urllib3.util.retry import Retry


class SenderClient:

    def __init__(self, url: str, auth_key: str):
        self.url = url
        self.session = self._get_session(auth_key)

    def get_mail_list(self, mail_list_slug: str) -> list[str]:
        return list(self._get_mail_list(mail_list_slug))

    def _get_mail_list(self, mail_list_slug: str) -> [str]:
        url = f'{self.url}/api/0/travel/maillist/{mail_list_slug}/content'
        while url:
            rsp = self.session.get(url)
            rsp.raise_for_status()
            data = rsp.json()
            for rec in data['result']:
                yield rec['email']
            url = data['next']

    @staticmethod
    def _get_session(auth_key: str):
        session = Session()
        retry = Retry(
            total=3,
            backoff_factor=2.0,
            status_forcelist=frozenset([500, 503, 413, 429])
        )
        session.mount('http://', HTTPAdapter(max_retries=retry))
        session.mount('https://', HTTPAdapter(max_retries=retry))
        session.auth = HTTPBasicAuth(auth_key, '')
        session.headers = dict()
        return session
