import logging
from datetime import datetime
from enum import Enum
from typing import Any

from parse import parse
from requests import Response, Session
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from tvmauth import TvmClient, TvmClientStatus, TvmApiClientSettings


class OrderType(Enum):
    AVIA_AEROFLOT = 'AVIA_AEROFLOT'
    HOTEL_BNOVO = 'HOTEL_BNOVO'
    HOTEL_DOLPHIN = 'HOTEL_DOLPHIN'
    HOTEL_EXPEDIA = 'HOTEL_EXPEDIA'
    HOTEL_TRAVELLINE = 'HOTEL_TRAVELLINE'
    SUBURBAN = 'SUBURBAN'
    TRAIN = 'TRAIN'


class TravelApiClient:

    __limit__ = 100

    def __init__(self, url: str, tvm_service_id: str, tvm_client_id: str, tvm_secret: str):
        self.url = url
        tvm_client = self._create_tvm_client(tvm_service_id, tvm_client_id, tvm_secret)
        self.tvm_ticket = tvm_client.get_service_ticket_for('cpa_export_api')
        self.session = self._get_session()

    def get_orders(
        self,
        order_type: OrderType,
        updated_at_from: datetime,
        updated_at_to: datetime,
        add_personal_data: bool,
    ) -> list[dict[str, Any]]:
        return list(self._get_orders(order_type, updated_at_from, updated_at_to, add_personal_data))

    def _get_orders(
        self,
        order_type: OrderType,
        updated_at_from: datetime,
        updated_at_to: datetime,
        add_personal_data: bool,
    ) -> [dict[str, Any]]:
        has_more = True
        while has_more:
            rsp = self._get_orders_page(order_type, updated_at_from, updated_at_to, add_personal_data)
            response = rsp.json()['order_snapshots']
            has_more = rsp.json()['has_more']
            if has_more:
                logging.info('Will get one more page')
                updated_at_from = self._parse_datetime_iso(response[-1]['updated_at'])
            yield from response

    @staticmethod
    def _create_tvm_client(tvm_service_id, tvm_client_id, tvm_secret):
        destinations = {'cpa_export_api': tvm_service_id}
        settings = TvmApiClientSettings(
            self_tvm_id=tvm_client_id,
            enable_service_ticket_checking=True,
            enable_user_ticket_checking=False,
            self_secret=tvm_secret,
            dsts=destinations,
        )
        client = TvmClient(settings)
        if client.status != TvmClientStatus.Ok:
            raise RuntimeError(f'Bad tvm client status: {client.status}')
        return client

    @staticmethod
    def _get_session():
        session = Session()
        retry = Retry(
            total=3,
            backoff_factor=2.0,
            status_forcelist=frozenset([500, 503, 413, 429])
        )
        session.mount('http://', HTTPAdapter(max_retries=retry))
        session.mount('https://', HTTPAdapter(max_retries=retry))
        return session

    def _get_orders_page(
        self,
        order_type: OrderType,
        updated_at_from: datetime,
        updated_at_to: datetime,
        add_personal_data: bool,
    ) -> Response:
        params = {
            'order_type': order_type.value,
            'updated_at_from_utc': updated_at_from.isoformat(),
            'updated_at_to_utc': updated_at_to.isoformat(),
            'add_personal_data': add_personal_data,
            'limit': self.__limit__,
        }
        headers = {}
        if self.tvm_ticket is not None:
            headers['X-Ya-Service-Ticket'] = self.tvm_ticket

        rsp = self.session.get(self.url, params=params, headers=headers)
        rsp.raise_for_status()
        return rsp

    @staticmethod
    def _parse_datetime_iso(dt_text: str) -> datetime:
        return parse('{:ti}', dt_text)[0]
