from datetime import datetime, timedelta, timezone

from travel.marketing.tools.lototron.order import Order
from travel.marketing.tools.lototron.travel_api_client import OrderType, TravelApiClient


class TravelApiOrders:

    __order_types__ = [
        OrderType.AVIA_AEROFLOT,
        OrderType.HOTEL_BNOVO,
        OrderType.HOTEL_DOLPHIN,
        OrderType.HOTEL_EXPEDIA,
        OrderType.HOTEL_TRAVELLINE,
        OrderType.TRAIN,
    ]

    def __init__(self, api_client: TravelApiClient, now: datetime, tz: timezone):
        self.api_client = api_client
        self.now = now
        self.tz = tz

    def get_event_orders(self, event_start: datetime, event_end: datetime) -> list[Order]:
        orders = self._get_all_type_orders(event_start, self.now + timedelta(hours=1))
        return self._filter_by_created_at(orders, event_start, event_end)

    def get_day_orders(self, orders: [Order], day_start: datetime, day_end: datetime) -> list[Order]:
        return self._filter_by_created_at(orders, day_start, day_end)

    def _get_all_type_orders(self, updated_at_from: datetime, updated_at_to: datetime) -> [Order]:
        order_ids = set()
        for order_type in self.__order_types__:
            for raw_order in self.api_client.get_orders(order_type, updated_at_from, updated_at_to, True):
                raw_order['order_type'] = order_type
                order = Order.from_raw_order(raw_order, self.tz)
                if order.partner_order_id in order_ids:
                    continue
                order_ids.add(order.partner_order_id)
                yield order

    @staticmethod
    def _filter_by_created_at(orders: [Order], created_at_from: datetime, created_at_to: datetime) -> list[Order]:
        def is_match(order: Order) -> bool:
            return created_at_from <= order.created_at < created_at_to
        return list(filter(is_match, orders))
