import logging

from argparse import Namespace
from dataclasses import asdict
from datetime import datetime, timezone

from email_validator import validate_email
from yt.wrapper import YtClient

from travel.hotels.lib.python3.yt.ytlib import ensure_table_exists, ypath_join, schema_from_dict
from travel.marketing.tools.lototron.event import Event
from travel.marketing.tools.lototron.order import HOTEL_ORDER_TYPES, Order, RawOrder
from travel.marketing.tools.lototron.sender_client import SenderClient
from travel.marketing.tools.lototron.travel_api_client import TravelApiClient
from travel.marketing.tools.lototron.travel_api_orders import TravelApiOrders


class WheelOfFortune:

    __wins_every__ = 20

    __orders_schema__ = schema_from_dict({
        'order_type': 'string',
        'partner_order_id': 'string',
        'created_at': 'string',
        'email': 'string',
        'email_hidden': 'string',
        'phone': 'string',
        'phone_hidden': 'string',
        'passport_id': 'string',
        'login': 'string',
        'order_amount': 'double',
        'points_to_topup': 'int64',
        'withdraw_points': 'int64',
    })

    def __init__(self, args: Namespace, tz: timezone):
        self.dump_path = args.dump_path
        self.sender_mail_list_slug = args.sender_mail_list_slug

        now = datetime.now(tz)

        self.event = Event(args, now, tz)

        self.api_client = TravelApiClient(
            url=args.travel_api_url,
            tvm_service_id=args.api_tvm_service_id,
            tvm_client_id=args.api_tvm_client_id,
            tvm_secret=args.api_tvm_secret,
        )

        self.sender_client = SenderClient(
            url=args.sender_url,
            auth_key=args.sender_auth_key,
        )

        self.yt_client = YtClient(args.yt_proxy, args.yt_token)

        self.api_orders = TravelApiOrders(
            api_client=self.api_client,
            now=now,
            tz=tz
        )

    def run(self):
        date_path = ypath_join(self.dump_path, str(self.event.yesterday_start.date()))

        participants = self._get_participants()
        logging.info(f'{len(participants)} participants')

        event_orders = self.api_orders.get_event_orders(self.event.event_start, self.event.event_end)
        logging.info(f'{len(event_orders)} event_orders')

        orders_to_win = self._get_orders_to_win(event_orders, participants)
        logging.info(f'{len(orders_to_win)} orders_to_win')
        self._dump_orders(orders_to_win, ypath_join(date_path, 'participant_orders'))

        won_orders = self._get_won_orders(orders_to_win)
        logging.info(f'{len(won_orders)} winners')

        self._dump_orders(won_orders, ypath_join(date_path, 'winners'))
        logging.info('All done')

    def _get_participants(self) -> set[str]:
        return {Order.get_normalized_email(e) for e in self.sender_client.get_mail_list(self.sender_mail_list_slug)}

    def _get_orders_to_win(self, orders: [Order], participants: set[str]) -> list[Order]:
        day_orders = self.api_orders.get_day_orders(orders, self.event.yesterday_start, self.event.yesterday_end)
        day_orders = sorted(day_orders, key=lambda x: (x.created_at, x.partner_order_id))
        emails = set()
        passport_ids = set()
        orders_to_win = list()
        for order in day_orders:
            if not order.email:
                logging.info(f'Skipping order. Has no email: {order.partner_order_id}')
                continue
            if order.order_status == 'CANCELLED':
                logging.info(f'Skipping order. Cancelled: {order.partner_order_id}')
                continue
            if order.order_status == 'REFUNDED':
                logging.info(f'Skipping order. Refunded: {order.partner_order_id}')
                continue
            if order.email not in participants:
                logging.info(f'Skipping order. Not participant: {order.partner_order_id}')
                continue
            if self._is_yandex_team_email(order.email):
                logging.info(f'Skipping order. Yandex-team email: {order.partner_order_id}')
                continue
            if order.order_type in HOTEL_ORDER_TYPES:
                # Getting user first hotel order only (among all user hotel orders made during bf action)
                if order.email in emails:
                    logging.info(f'Skipping order. Repeated by email: {order.partner_order_id}')
                    continue
                if order.passport_id and order.passport_id in passport_ids:
                    logging.info(f'Skipping order. Repeated by passport_id: {order.partner_order_id}')
                    continue
                emails.add(order.email)
                passport_ids.add(order.passport_id)
            orders_to_win.append(order)
        return orders_to_win

    def _get_won_orders(self, orders: list[Order]) -> list[Order]:
        return orders[self.__wins_every__ - 1::self.__wins_every__]

    def _dump_orders(self, orders: [Order], table_path: str) -> None:
        if self.yt_client.exists(table_path):
            raise Exception(f'Table already exists {table_path}')
        ensure_table_exists(table_path, self.yt_client, self.__orders_schema__)
        data = (self._order_to_dict(o) for o in orders)
        self.yt_client.write_table(table_path, data)

    @staticmethod
    def _order_to_dict(order: Order) -> RawOrder:
        d = asdict(order)
        d.pop('order_status', None)
        d.pop('topup_time', None)
        overrides = dict(order_type=order.order_type.value, created_at=str(order.created_at))
        return {**d, **overrides}

    @staticmethod
    def _is_yandex_team_email(email: str):
        email = validate_email(email, check_deliverability=False)
        return email.ascii_domain == 'yandex-team.ru'
