import logging
import sys
from argparse import Namespace, ArgumentParser
from datetime import datetime, timezone, timedelta
from typing import Any, Optional

from yt.wrapper import YtClient
from yql.api.v1.client import YqlClient

from travel.library.python.tools import replace_args_from_env
from travel.hotels.lib.python3.yql import yqllib
from travel.hotels.lib.python3.yt.ytlib import ensure_table_exists, schema_from_dict, ypath_join
from travel.hotels.lib.python3.yt.versioned_path import KeepLastNCleanupStrategy


class NotFinishedSearchesNotifierTrains:

    __timezone__ = timezone(timedelta(hours=3))
    __tz_name__ = 'Europe/Moscow'
    __time_format__ = '%Y-%m-%d %H:%M:%S'

    __result_table_schema__ = schema_from_dict({
        'email': 'string',
        'event': 'any',
    })

    def __init__(self, args: Namespace):
        self.args = args
        self.yql_client = YqlClient(db=args.yt_proxy, token=args.yql_token)
        self.yt_client = YtClient(args.yt_proxy, args.yt_token)

    def run(self) -> None:
        if not list(self.yt_client.list(self.args.results_path)):
            # Make Range happy
            table_name = datetime.now(tz=self.__timezone__).strftime(self.__time_format__)
            table_path = ypath_join(self.args.results_path, table_name)
            logging.info(f'Creating dummy table {table_path}')
            ensure_table_exists(table_path, self.yt_client, self.__result_table_schema__)
        query_args = {
            '$logs_path': self.args.logs_path,
            '$orders_table': self.args.orders_table,
            '$no_spam_after_buy_period': self.args.no_spam_after_buy_period,
            '$results_path': self.args.results_path,
        }
        self._run_query('get_searches.yql', query_args)
        logging.info('Processing finished')

        cleaner = KeepLastNCleanupStrategy(self.args.tables_to_keep_count)
        cleaner.clean(self.args.results_path, self.yt_client)
        logging.info('All done')

    def _run_query(self, query_name: str, query_args: dict[str, Any], transaction_id: Optional[str] = None) -> None:
        logging.info(f'Running query {query_name}, Args: {query_args}')
        yqllib.run_yql_file(
            client=self.yql_client,
            resource_name=query_name,
            project_name=self.__class__.__name__,
            parameters=query_args,
            transaction_id=transaction_id,
        )
        logging.info('Query finished')


def main():
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)-15s | %(module)s | %(levelname)s | %(message)s',
        stream=sys.stdout,
    )

    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yql-token', required=True)
    parser.add_argument(
        '--logs-path',
        default='//home/logfeller/logs/travel-trains-search_api-search_log-production/1h',
    )
    parser.add_argument('--orders-table', default='//home/travel/prod/cpa/train/orders')
    parser.add_argument('--no-spam-after-buy-period', type=int, default=7)
    parser.add_argument('--results-path', required=True)
    parser.add_argument('--tables-to-keep-count', type=int, default=50)
    args = parser.parse_args(args=replace_args_from_env())

    NotFinishedSearchesNotifierTrains(args).run()


if __name__ == '__main__':
    main()
