package main

import (
	"context"
	"fmt"

	"github.com/jonboulle/clockwork"
	"google.golang.org/grpc"
	ghealth "google.golang.org/grpc/health"
	"google.golang.org/grpc/health/grpc_health_v1"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/containers"
	dynamicdicts "a.yandex-team.ru/travel/library/go/dicts/updaterservice"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	"a.yandex-team.ru/travel/library/go/grpcutil/service/servingstatus"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
	"a.yandex-team.ru/travel/notifier/internal/api"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/dicts"
	"a.yandex-team.ru/travel/notifier/internal/extractors"
	mailsenderhandler "a.yandex-team.ru/travel/notifier/internal/mailsender/handler"
	"a.yandex-team.ru/travel/notifier/internal/models"
	orderchangeshandler "a.yandex-team.ru/travel/notifier/internal/orderchanges/handler"
	pullnotificationsgrpc "a.yandex-team.ru/travel/notifier/internal/pullnotifications/grpc"
	"a.yandex-team.ru/travel/notifier/internal/rollout"
	"a.yandex-team.ru/travel/notifier/internal/service/onlineregistration"
	"a.yandex-team.ru/travel/notifier/internal/service/orderchanges"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip"
	pretripscheduling "a.yandex-team.ru/travel/notifier/internal/service/pretrip/scheduling"
	"a.yandex-team.ru/travel/notifier/internal/service/profiling"
	"a.yandex-team.ru/travel/notifier/internal/service/pullnotifications"
	"a.yandex-team.ru/travel/notifier/internal/service/readiness"
	rolloutservice "a.yandex-team.ru/travel/notifier/internal/service/rollout"
	"a.yandex-team.ru/travel/notifier/internal/service/scheduler"
	"a.yandex-team.ru/travel/notifier/internal/service/shutdown"
	"a.yandex-team.ru/travel/notifier/internal/service/subscriptions"
	"a.yandex-team.ru/travel/notifier/internal/service/subscriptions/logging/useractions"
	subscriptionshandler "a.yandex-team.ru/travel/notifier/internal/subscriptions/handler"
	"a.yandex-team.ru/travel/notifier/internal/ytdicts"
)

const (
	serviceName = "travel-notifier"
)

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &api.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.NewDeploy(&api.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
		}
		ctxCancel()
	}()

	tvmAllowedIds := tvmutil.TvmClientIDFromInt(api.Cfg.Tvm.WhiteList)
	tvmClient, err := tvmtool.NewDeployClient()
	if err != nil {
		logger.Errorf("failed to create tvm client: %s", err)
	}

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, api.Cfg.Metrics.PerfMetricsRefreshInterval)

	pgClient, err := api.BuildPGClient(logger)
	if err != nil {
		logger.Errorf("failed to create PG client: %s", err)
		return
	}

	if err := models.MigrateAndInit(pgClient); err != nil {
		logger.Errorf(err.Error())
		return
	}

	clock := clockwork.NewRealClock()

	recipientsRepository := database.NewRecipientsRepository(pgClient)
	usersRepository := database.NewUsersRepository(pgClient)
	ordersRepository := database.NewOrdersRepository(pgClient)
	notificationsRepository := api.BuildNotificationsRepository(pgClient)

	senderClient := api.BuildSenderClient(logger)

	unifiedAgentClient, err := unifiedagent.NewGrpcClient(&api.Cfg.UnifiedAgent, logger, nil)
	if err != nil {
		logger.Error("failed to create unified agent client", log.Error(err))
		return
	}

	ytDictsRegistry := ytdicts.NewRegistry(api.Cfg.YtDicts, logger.WithName("YtDictsRegistry"))
	err = ytDictsRegistry.UpdateRegistryOnce(ctx)
	if err != nil {
		logger.Error("YT registry pre-start check failed", log.Error(err))
		if !api.Cfg.YtDicts.IgnoreStartupErrors {
			return
		}
	}
	ytDictsRegistry.BackgroundRun()

	ordersClient := api.BuildOrdersClient(logger)
	if ordersClient == nil {
		logger.Error("failed to create a client to the orders app")
		return
	}

	notificationsScheduler := scheduler.NewService(logger, notificationsRepository, recipientsRepository)

	betterPriceSubscriptionRepository := api.BuildBetterPriceSubscriptionRepository(pgClient)
	subscriptionsService := subscriptions.NewService(
		logger,
		recipientsRepository,
		betterPriceSubscriptionRepository,
		subscriptions.NewMD5UnsubscribeHashGenerator(
			api.Cfg.Subscriptions.UnsubscribeHashSecret,
			&subscriptions.RandSaltProvider{},
			api.Cfg.Subscriptions.SaltLength,
		),
		senderClient,
		api.Cfg.Subscriptions,
		clock,
		useractions.NewLogger(unifiedAgentClient),
		ytDictsRegistry.GetPromoEventsRepository(),
		ordersClient,
		notificationsScheduler,
	)
	grpcSubscriptionsService := subscriptionshandler.NewGRPCSubscriptionsHandler(
		logger,
		subscriptionsService,
		subscriptions.NewBetterPriceSubscriptionMapper(),
	)

	templateMapper, err := api.Cfg.MailSender.NewTemplateSenderMapper(senderClient, nil)
	// TODO(mikhailche): pass renderer client when implemented
	if err != nil {
		logger.Error("failed to create template mapper", log.Error(err))
	}
	mailSenderService := mailsenderhandler.NewGRPCMailSenderHandler(
		logger,
		templateMapper,
	)
	dictsRegistry, err := dicts.NewRegistry(api.Cfg.Dicts, logger)
	if err != nil {
		logger.Error("failed to create dicts registry", log.Error(err))
		return
	}
	pretripNotificationsBuilder := pretripscheduling.NewNotificationsBuilder(
		pretripscheduling.NotificationsBuilderConfig{
			IsTesting:                 api.Cfg.IsTesting(),
			TestingEmails:             containers.SetOf(api.Cfg.Pretrip.Testing.Emails...),
			AdhocSendingInterval:      api.Cfg.Pretrip.Testing.AdhocSendingInterval,
			WeekBeforeSendingInterval: api.Cfg.Pretrip.Testing.WeekBeforeSendingInterval,
			DayBeforeSendingInterval:  api.Cfg.Pretrip.Testing.DayBeforeSendingInterval,
		},
	)
	rollOutService := rolloutservice.NewService(api.Cfg.RollOut)

	settlementDataProvider := extractors.NewSettlementDataProvider(
		dictsRegistry.GetSettlementsRepository(),
		dictsRegistry.GetTimeZonesRepository(),
	)

	stationDataProvider := extractors.NewStationDataProvider(
		dictsRegistry.GetStationsRepository(),
		dictsRegistry.GetStationToSettlementRepository(),
		dictsRegistry.GetSettlementsRepository(),
		dictsRegistry.GetStationCodesRepository(),
	)

	pretripNotificationScheduler := pretrip.NewPretripNotificationsService(
		logger,
		notificationsScheduler,
		pretripNotificationsBuilder,
		recipientsRepository,
		api.BuildOrderDestinationExtractor(dictsRegistry),
		api.Cfg.Pretrip,
		clock,
		rollOutService,
		rollout.NewEmailFilter(api.Cfg.Pretrip.EmailRules),
		settlementDataProvider,
		stationDataProvider,
	)

	onlineRegistrationNotificationScheduler := onlineregistration.NewNotificationsService(
		api.Cfg.OnlineRegistration,
		logger,
		notificationsScheduler,
		usersRepository,
		onlineregistration.NewNotificationsBuilder(
			dictsRegistry.GetStationCodesRepository(),
			dictsRegistry.GetStationsRepository(),
			dictsRegistry.GetTimeZonesRepository(),
		),
		clock,
	)
	orderChangesService := orderchanges.NewService(
		logger,
		api.Cfg.HandleChangesFromQueue,
		ordersClient,
		ordersRepository,
		pretrip.NewOrderStatusChecker(ordersRepository),
		pretripNotificationScheduler,
		onlineRegistrationNotificationScheduler,
	)
	orderChangesHandler := orderchangeshandler.NewGRPCOrderChangesHandler(logger, orderChangesService.OnOrderChanged)

	pullNotificationsService := pullnotifications.NewService(clock, notificationsRepository)
	pullNotificationsHandler := pullnotificationsgrpc.NewService(pullNotificationsService)

	go func() {
		err = metricserver.RunMetricsHTTPServer(context.Background(), api.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	grpcServiceRegisterers := []grpcserver.ServiceRegisterer{
		orderChangesHandler.GetServiceRegisterer(),
		grpcSubscriptionsService.GetServiceRegisterer(),
		mailSenderService.GetServiceRegisterer(),
		pullNotificationsHandler.GetServiceRegisterer(),
	}

	healthServer := ghealth.NewServer()
	grpcServer := grpcserver.NewDefaultGrpcServerBuilder(
		api.Cfg.Grpc,
		grpcServiceRegisterers,
		logger.WithName("GRPCServer").(*zap.Logger),
		tvmClient,
		tvmAllowedIds,
		rootRegistry.WithPrefix("grpc"),
	).WithHealthServer(healthServer).Build()

	waitShutdown := runShutdownServer(logger, healthServer, grpcServer, ctxCancel)
	runReadinessServer(ctx, logger)
	runProfilingService()
	runDynamicResourcesService(logger, dictsRegistry)

	servingStatusService := servingstatus.NewService(
		logger,
		serviceName,
		healthServer.SetServingStatus,
		api.Cfg.HealthCheck.UpdateInterval,
		clockwork.NewRealClock(),
	).Requires(
		func() bool {
			_, err := pgClient.GetPrimary()
			return err == nil
		},
	).Requires(
		func() bool {
			return ytDictsRegistry.AllReady()
		},
	)
	servingStatusService.MonitorServingStatus(ctx)

	go func() {
		err = grpcServer.Run(context.Background())
		if err != nil {
			logger.Fatal("Error while starting Grpc server", log.Error(err))
		}
	}()

	waitShutdown()
	logger.Info("the instance has been shut down")
}

func runShutdownServer(
	logger *zap.Logger,
	healthServer *ghealth.Server,
	grpcServer *grpcserver.GrpcServer,
	ctxCancel context.CancelFunc,
) (waitShutdown func()) {
	shutdownService := shutdown.NewService(
		api.Cfg.Shutdown,
		func() {
			logger.Info("the instance is going to be shut down")
			healthServer.Shutdown()
			grpcServer.GracefulStop()
			ctxCancel()
		},
	)
	return shutdownService.BackroundRun()
}

func runReadinessServer(ctx context.Context, logger *zap.Logger) {
	tvmHelper := tvmutil.NewDeployTvmHelper(
		logger,
		&tvmutil.TvmHelperConfig{
			SelfID:    api.Cfg.Tvm.SelfAppID,
			WhiteList: []uint32{},
		},
	)
	tvmInterceptor := tvmHelper.GRPCClientInterceptor(api.Cfg.Tvm.SelfAppID)
	if tvmInterceptor == nil {
		logger.Error("failed to create tvm interceptor for readiness server")
		return
	}
	readinessService := readiness.NewService(
		api.Cfg.Readiness,
		func() bool {
			requestCtx, cancel := context.WithTimeout(ctx, api.Cfg.Readiness.Timeout)
			defer cancel()
			clientConn, err := grpc.DialContext(
				requestCtx,
				api.Cfg.Grpc.Addr,
				grpc.WithInsecure(),
				grpc.WithBlock(),
				grpc.WithUnaryInterceptor(tvmInterceptor),
			)
			if err != nil {
				return false
			}
			defer clientConn.Close()
			response, err := grpc_health_v1.NewHealthClient(clientConn).Check(
				requestCtx,
				&grpc_health_v1.HealthCheckRequest{Service: serviceName},
			)
			return err == nil && response.GetStatus() == grpc_health_v1.HealthCheckResponse_SERVING
		},
	)
	readinessService.BackroundRun()
}

func runProfilingService() {
	profilingService := profiling.NewService(api.Cfg.Profiling)
	profilingService.BackroundRun()
}

func runDynamicResourcesService(logger *zap.Logger, dictsRegistry *dicts.Registry) {
	dynamicResourceService := dynamicdicts.NewService(
		logger,
		api.Cfg.DynamicResources,
		dictsRegistry.OnUpdate,
	)
	dynamicResourceService.BackroundRun()
}
