package main

import (
	"context"
	"fmt"
	"time"

	"github.com/jonboulle/clockwork"
	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/grpcutil/client/ypresolver"
	"a.yandex-team.ru/travel/library/go/logbroker"
	multilogbroker "a.yandex-team.ru/travel/library/go/logbroker/multi"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/services/messagecollector"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
	"a.yandex-team.ru/travel/notifier/internal/collector"
	"a.yandex-team.ru/travel/notifier/internal/collector/orderchanges"
	"a.yandex-team.ru/travel/notifier/internal/collector/unprocessed"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/processor"
	"a.yandex-team.ru/travel/notifier/internal/service/profiling"
	"a.yandex-team.ru/travel/notifier/internal/service/readiness"
	"a.yandex-team.ru/travel/notifier/internal/service/shutdown"
)

const (
	serviceName = "travel-notifier-collector"
)

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &collector.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.NewDeploy(&collector.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
		}
		ctxCancel()
	}()

	_, err = tvmtool.NewDeployClient()
	if err != nil {
		logger.Errorf("failed to create tvm client: %s", err)
	}

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, collector.Cfg.Metrics.PerfMetricsRefreshInterval)

	go func() {
		err = metricserver.RunMetricsHTTPServer(ctx, processor.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	var unprocessedOrdersLogbrokerProducer = createUnprocessedOrderLogbrokerProducer(ctx, logger)
	defer unprocessedOrdersLogbrokerProducer.Close()
	clock := clockwork.NewRealClock()
	pgClient, err := collector.BuildPGClient(logger)
	if err != nil {
		logger.Errorf("failed to create PG client: %s", err)
		return
	}

	unprocessedOrdersStorage := database.NewUnprocessedOrdersStorage(pgClient)
	unprocessedOrdersService := unprocessed.NewService(
		logger,
		unprocessedOrdersLogbrokerProducer,
		clock,
		unprocessedOrdersStorage,
	)
	go unprocessedOrdersService.MonitorFailedOrdersCount()
	tvmHelper := tvmutil.NewDeployTvmHelper(
		logger,
		&tvmutil.TvmHelperConfig{
			SelfID: collector.Cfg.Tvm.SelfAppID,
		},
	)
	connection, err := createConnection(
		collector.Cfg.OrderChangesService.YPlannerID,
		collector.Cfg.OrderChangesService.TvmID,
		logger,
		&tvmHelper,
	)
	if err != nil {
		logger.Fatal("failed to create connection to processor", log.Error(err))
	}
	defer func() {
		if err := connection.Close(); err != nil {
			logger.Error("failed to close connection to processor", log.Error(err))
		}
	}()

	orderChangesService := orderchanges.NewService(connection, collector.Cfg.OrderChangesService)
	collectorService := collector.BuildCollector(logger, collector.Cfg, collector.Cfg.Logbroker, orderChangesService, unprocessedOrdersService)
	go func() {
		collectorService.Run(ctx)
	}()

	waitShutdown := runShutdownServer(logger, ctxCancel)
	runReadinessServer(collectorService)
	runProfilingService()

	waitShutdown()
	logger.Info("the instance has been shut down")
}

func createUnprocessedOrderLogbrokerProducer(ctx context.Context, logger log.Logger) unprocessed.LogbrokerProducer {
	if collector.Cfg.UnprocessedService.Mock {
		return unprocessed.NewMockProducer()
	}
	logbrokerProducer, err := logbroker.NewProducer(
		collector.Cfg.UnprocessedService.Topic,
		collector.Cfg.UnprocessedService.Endpoint,
		multilogbroker.GetDeploySourceID(collector.Cfg.UnprocessedService.ProducerID),
		logbroker.NewOAuthCredentialsProvider(collector.Cfg.UnprocessedService.Token),
		logger,
		logbroker.WithoutSeqNo(),
	)
	if err != nil {
		logger.Fatal("failed to create logbroker producer", log.Error(err))
	}
	err = logbrokerProducer.Run(ctx)
	if err != nil {
		logger.Fatal("failed to run logbroker producer", log.Error(err))
	}
	return logbrokerProducer
}

func runShutdownServer(
	logger *zap.Logger,
	ctxCancel context.CancelFunc,
) (waitShutdown func()) {
	shutdownService := shutdown.NewService(
		collector.Cfg.Shutdown,
		func() {
			logger.Info("the instance is going to be shut down")
			ctxCancel()
		},
	)
	return shutdownService.BackroundRun()
}

func runReadinessServer(collectorService *messagecollector.Collector) {
	readinessService := readiness.NewService(
		collector.Cfg.Readiness,
		collectorService.Ready,
	)
	readinessService.BackroundRun()
}

func runProfilingService() {
	profilingService := profiling.NewService(processor.Cfg.Profiling)
	profilingService.BackroundRun()
}

func createConnection(yPlannerID string, tvmID uint32, logger log.Logger, tvmHelper *tvmutil.TvmHelper) (*grpc.ClientConn, error) {
	tvmInterceptor := tvmHelper.GRPCClientInterceptor(tvmID)
	if tvmInterceptor == nil {
		return nil, fmt.Errorf("failed to create tvm interceptor for readiness server")
	}
	ctx := context.Background()
	ctx, cancel := context.WithTimeout(ctx, 90*time.Second)
	defer cancel()
	resolver := ypresolver.NewYPResolverBuilder(ypresolver.WithLogger(logger.Structured()))
	return grpc.DialContext(
		ctx,
		ypresolver.BuildServiceFQDN(yPlannerID),
		grpc.WithResolvers(resolver),
		grpc.WithDefaultServiceConfig("{\"loadBalancingPolicy\":\"healthcheck_balancer\"}"),
		grpc.WithInsecure(),
		grpc.WithBlock(),
		grpc.WithUnaryInterceptor(tvmInterceptor),
	)
}
