package main

import (
	"context"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	contentadminlib "a.yandex-team.ru/travel/library/go/contentadmin"
	contentadminapi "a.yandex-team.ru/travel/marketing/content/v1"
	"a.yandex-team.ru/travel/notifier/internal/contentadmin"
	"a.yandex-team.ru/travel/notifier/internal/externalhttp"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/audioguide"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/events"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/hotels"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/weather"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/structs"
)

type CachingContentAdmin struct {
	realClient   interfaces.CityImagesProvider
	cachedResult map[int]*ContentAdminResponse
}

type ContentAdminResponse struct {
	payload *contentadminapi.CityImage
	err     error
}

func (c *CachingContentAdmin) GetCityImageByID(ctx context.Context, geoID int) (*contentadminapi.CityImage, error) {
	result, ok := c.cachedResult[geoID]
	if !ok {
		payload, err := c.realClient.GetCityImageByID(ctx, geoID)
		result = &ContentAdminResponse{payload, err}
		c.cachedResult[geoID] = result
	}
	return result.payload, result.err
}

func NewContentAdminClient(
	logger log.Logger, selfTvmID uint32, contacts contentadmin.Config,
) interfaces.CityImagesProvider {
	return &CachingContentAdmin{
		realClient:   contentadminlib.NewContentAdminClient(logger, selfTvmID, contacts.ToLibConfig()),
		cachedResult: map[int]*ContentAdminResponse{},
	}
}

type CachingIziTravelClient struct {
	realClient   interfaces.AudioGuidesProvider
	cachedResult *IziTravelClientResponse
}

type IziTravelClientResponse struct {
	payload structs.AudioGuidesForCity
	err     error
}

func (c *CachingIziTravelClient) GetAudioGuides(ctx context.Context, geoID int) (structs.AudioGuidesForCity, error) {
	if c.cachedResult == nil {
		payload, err := c.realClient.GetAudioGuides(ctx, geoID)
		c.cachedResult = &IziTravelClientResponse{payload, err}
	}
	return c.cachedResult.payload, c.cachedResult.err
}

func NewAudioGuidesProvider(httpClient *externalhttp.HTTPClient) interfaces.AudioGuidesProvider {
	return &CachingIziTravelClient{realClient: audioguide.NewAudioGuidesProvider(httpClient)}
}

type CachingAfishaClient struct {
	realClient   interfaces.EventsProvider
	cachedResult *AfishaClientResponse
}

type AfishaClientResponse struct {
	payload structs.EventsForCity
	err     error
}

func (c *CachingAfishaClient) GetEvents(
	ctx context.Context,
	geoID int,
	startDate time.Time,
	latitude float64,
	longitude float64,
) (structs.EventsForCity, error) {
	if c.cachedResult == nil {
		payload, err := c.realClient.GetEvents(ctx, geoID, time.Now().AddDate(0, 0, 2), latitude, longitude)
		c.cachedResult = &AfishaClientResponse{payload, err}
	}
	return c.cachedResult.payload, c.cachedResult.err
}

func NewAfishaEventsClient(httpClient *externalhttp.HTTPClient) interfaces.EventsProvider {
	return &CachingAfishaClient{realClient: events.NewAfishaEventsClient(httpClient)}
}

type CachingWeatherClient struct {
	realClient   interfaces.WeatherProvider
	cachedResult map[int]*WeatherClientResponse
}

type WeatherClientResponse struct {
	payload structs.WeatherForCity
	err     error
}

func (c *CachingWeatherClient) GetWeather(ctx context.Context, geoID int, lang string) (structs.WeatherForCity, error) {
	result, ok := c.cachedResult[geoID]
	if !ok {
		payload, err := c.realClient.GetWeather(ctx, geoID, "ru")
		result = &WeatherClientResponse{payload, err}
		c.cachedResult[geoID] = result
		time.Sleep(200 * time.Millisecond) // limit the load we generate for the weather site to 5rps
	}
	return result.payload, result.err
}

func NewWeatherProvider(httpClient *externalhttp.HTTPClient) interfaces.WeatherProvider {
	return &CachingWeatherClient{
		realClient:   weather.NewWeatherProvider(httpClient),
		cachedResult: map[int]*WeatherClientResponse{},
	}
}

type CachingHotelsClient struct {
	realClient    interfaces.HotelsProvider
	cached        map[int]HotelClientResponse
	sampleOrderID string
}

type HotelClientResponse struct {
	payload *structs.HotelPayload
	err     error
}

func (c *CachingHotelsClient) GetHotels(
	ctx context.Context,
	geoID int,
	orderID string,
	limit int,
) (*structs.HotelPayload, error) {
	result, ok := c.cached[geoID]
	if !ok {
		payload, err := c.realClient.GetHotels(ctx, geoID, c.sampleOrderID, limit)
		c.cached[geoID] = HotelClientResponse{payload, err}
		time.Sleep(200 * time.Millisecond) // limit the load we generate for the hotels site to 5rps
		return payload, err
	}
	return result.payload, result.err
}

func NewHotelsClient(httpClient *externalhttp.HTTPClient, sampleOrderID string) interfaces.HotelsProvider {
	return &CachingHotelsClient{
		realClient:    hotels.NewHotelsClient(httpClient),
		cached:        make(map[int]HotelClientResponse),
		sampleOrderID: sampleOrderID,
	}
}
