package main

import (
	"context"
	"fmt"
	"net/http"
	"reflect"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/notifier/cmd/localblocks/blocktests"
	"a.yandex-team.ru/travel/notifier/internal/dicts"
	"a.yandex-team.ru/travel/notifier/internal/externalhttp"
	"a.yandex-team.ru/travel/notifier/internal/extractors"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/processor"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/ui"
	"a.yandex-team.ru/travel/notifier/internal/service/subscriptions"
	"a.yandex-team.ru/travel/notifier/internal/travelapi"
)

const (
	serviceName          = "travel-notifier-api"
	mandatoryEmailSuffix = "@yandex-team.ru"
)

type ExtraParams struct {
	SampleOrderID string `config:"SAMPLE_ORDER_ID,required"`
}

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &processor.Cfg)
	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	extraParams := ExtraParams{}
	err = config.Load(ctx, &extraParams)
	if err != nil {
		fmt.Println("can not load extra params:", err)
		ctxCancel()
		return
	}

	logger, err := logging.NewDeploy(&processor.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	defer func() {
		ctxCancel()
	}()

	tvmClient, err := tvmtool.NewDeployClient()
	if err != nil {
		logger.Errorf("failed to create tvm client: %s", err)
	}

	ordersClient := processor.BuildOrdersClient(logger)
	if ordersClient == nil {
		logger.Error("failed to create a client to the orders app")
		return
	}

	dictsRegistry, err := dicts.NewRegistry(processor.Cfg.Dicts, logger)
	if err != nil {
		logger.Error("failed to create dicts registry", log.Error(err))
		return
	}

	stationCodeSeeker := newStationCodeSeeker(dictsRegistry)

	contentAdminClient := NewContentAdminClient(logger, processor.Cfg.Tvm.SelfAppID, processor.Cfg.ContentAdmin)
	if contentAdminClient == nil {
		logger.Error("failed to create a client to the content admin service")
		return
	}

	iziTravelClient := NewAudioGuidesProvider(
		externalhttp.NewHTTPClient(
			processor.Cfg.Pretrip.BlockProviders.AudioGuidesProvider,
			&http.Client{
				Timeout: processor.Cfg.Pretrip.BlockProviders.AudioGuidesProviderTimeout,
			},
			tvmClient,
			externalhttp.WithRequestTimeout(processor.Cfg.Pretrip.BlockProviders.AudioGuidesProviderTimeout),
		),
	)

	afishaClient := NewAfishaEventsClient(
		externalhttp.NewHTTPClient(
			processor.Cfg.Pretrip.BlockProviders.EventsProvider,
			&http.Client{
				Timeout: processor.Cfg.Pretrip.BlockProviders.EventsProviderTimeout,
			},
			tvmClient,
			externalhttp.WithRequestTimeout(processor.Cfg.Pretrip.BlockProviders.EventsProviderTimeout),
		),
	)

	weatherClient := NewWeatherProvider(
		externalhttp.NewHTTPClient(
			processor.Cfg.Pretrip.BlockProviders.WeatherProvider,
			&http.Client{
				Timeout: processor.Cfg.Pretrip.BlockProviders.WeatherProviderTimeout,
			},
			tvmClient,
			externalhttp.WithRequestTimeout(processor.Cfg.Pretrip.BlockProviders.WeatherProviderTimeout),
		),
	)

	hotelsClient := NewHotelsClient(
		externalhttp.NewHTTPClient(
			processor.Cfg.Pretrip.BlockProviders.HotelsProvider,
			&http.Client{
				Timeout: processor.Cfg.Pretrip.BlockProviders.HotelsProviderTimeout,
			},
			tvmClient,
			externalhttp.WithRequestTimeout(processor.Cfg.Pretrip.BlockProviders.HotelsProviderTimeout),
		),
		extraParams.SampleOrderID, //"b4526782-9230-496a-8361-ccce10d4f148",
	)
	additionalOrderInfoClient := travelapi.NewAdditionalOrderInfoClient(
		externalhttp.NewHTTPClient(
			processor.Cfg.Pretrip.BlockProviders.AdditionalOrderInfo,
			&http.Client{
				Timeout: processor.Cfg.Pretrip.BlockProviders.AdditionalOrderInfoProviderTimeout,
			},
			tvmClient,
			externalhttp.WithRequestTimeout(processor.Cfg.Pretrip.BlockProviders.AdditionalOrderInfoProviderTimeout),
		),
	)

	orderDestinationExtractor := processor.BuildOrderDestinationExtractor(dictsRegistry)
	unsubscribeLinkGenerator := subscriptions.NewUnsubscribeLinkGenerator(
		processor.Cfg.TravelPortalURL,
		processor.Cfg.Subscriptions.UnsubscribePath,
	)
	settlementDataProvider := extractors.NewSettlementDataProvider(
		dictsRegistry.GetSettlementsRepository(),
		dictsRegistry.GetTimeZonesRepository(),
	)
	stationDataProvider := extractors.NewStationDataProvider(
		dictsRegistry.GetStationsRepository(),
		dictsRegistry.GetStationToSettlementRepository(),
		dictsRegistry.GetSettlementsRepository(),
		dictsRegistry.GetStationCodesRepository(),
	)

	pretripBlocksCollector := blocks.NewBlocksCollector(
		logger,
		pretrip.GetDefaultBlockProviders(
			processor.Cfg.TravelPortalURL,
			contentAdminClient,
			iziTravelClient,
			afishaClient,
			weatherClient,
			orderDestinationExtractor,
			unsubscribeLinkGenerator,
			settlementDataProvider,
			stationDataProvider,
			dictsRegistry.GetTimeZonesRepository(),
			hotelsClient,
			additionalOrderInfoClient,
		)...,
	)

	type subtypeTestConfig struct {
		startDate int // relative to today, i.e. 7 means "a week after today"
		subtype   models.NotificationSubtype
	}

	type orderTypeTestConfig struct {
		arrivalDate int // relative to startDate, i.e. +1 means "next day after a start date"
		orderType   models.OrderType
	}

	orderTypesMap := map[models.OrderType]orders.OrderType{
		models.OrderHotel: orders.OrderTypeHotel,
		models.OrderTrain: orders.OrderTypeTrain,
	}

	variantsCount := 0
	errorsCount := map[string]int{}
	errorsCache := map[string]bool{}
	// loop over 80 cities
	for _, settlement := range processor.Cfg.Pretrip.SettlementsWhitelist {
		geoID, found := settlementDataProvider.GetGeoID(settlement)
		if !found {
			logger.Error("got no geoID for settlement", log.Int("settlement", settlement))
			registerError(errorsCount, nil, "any order", "any subtype")
			continue
		}
		seekStationExpressCode, err := stationCodeSeeker.seekStationExpressCode(settlement)
		if err != nil {
			logger.Info(err.Error(), log.Int("settlement", settlement))
			seekStationExpressCode = ""
		}

		orderTypeTestConfigs := []orderTypeTestConfig{
			{0, models.OrderHotel},
			{1, models.OrderTrain}, // overnight train
		}
		for _, orderTypeConfig := range orderTypeTestConfigs {
			subtypeTestConfigs := []subtypeTestConfig{
				{7, models.NotificationWeekBefore},
				{1, models.NotificationDayBefore},
				{5, models.NotificationAdhoc},
			}
			if seekStationExpressCode == "" && orderTypeConfig.orderType == models.OrderTrain {
				// no express station code for this settlement - no way to test the train order to it
				continue
			}
			unsubscribeHash := "test-unsubscribe-hash-123"
			ru := "ru"
			for _, subtypeConfig := range subtypeTestConfigs {
				variantsCount++
				testOrderID := "test-order-id"
				departureTime := time.Now().AddDate(0, 0, subtypeConfig.startDate)
				checkInTime := departureTime
				arrivalTime := departureTime.AddDate(0, 0, orderTypeConfig.arrivalDate)
				orderInfo := orders.OrderInfo{
					ID:   testOrderID,
					Type: orderTypesMap[orderTypeConfig.orderType],
					TrainOrderItems: []*orders.TrainOrderItem{
						&orders.TrainOrderItem{
							ArrivalStation: seekStationExpressCode,
							DepartureTime:  &departureTime,
							ArrivalTime:    &arrivalTime,
						},
					},
					HotelOrderItems: []*orders.HotelOrderItem{
						{
							CheckInDate: &checkInTime,
							GeoRegions: []*orders.GeoRegion{
								{
									GeoID: int64(geoID),
									Type:  6, // geoIDTypeSettlement
								},
							},
						},
					},
				}
				notification := models.Notification{
					ID:          999999,
					Status:      models.NotificationStatusPlanned,
					RecipientID: ptr.Int32(999999),
					Recipient: &models.Recipient{
						ID:              999999,
						Email:           ptr.String("test-recipient@yandex-team.ru"),
						IsSubscribed:    true,
						UnsubscribeHash: &unsubscribeHash,
						NationalVersion: &ru,
						Language:        &ru,
					},
					TypeID: 1,
					Type: models.NotificationType{
						ID:   1,
						Name: models.NotificationTypePretrip.Name,
					},
					Subtype:   subtypeConfig.subtype,
					ChannelID: 1,
					Channel:   models.NotificationChannelEmail,
					OrderID:   &testOrderID,
					Order: &models.Order{
						ID:   testOrderID,
						Type: orderTypeConfig.orderType,
					},
					Failures:     0,
					DispatchType: models.DispatchTypePush,
				}
				blockByType, err := pretripBlocksCollector.GetBlocks(
					ctx,
					processor.Cfg.Pretrip.NotificationConfigByType[orderTypeConfig.orderType].Blocks,
					&orderInfo,
					notification,
				)
				if err != nil {
					if _, ok := errorsCache[err.Error()]; !ok {
						logger.Error("got error", log.Error(err))
						errorsCache[err.Error()] = true
					}
					registerError(errorsCount, nil, orderTypeConfig.orderType, subtypeConfig.subtype)
					continue
				}
				if len(blockByType) == 0 {
					logger.Error(
						"got no blocks",
						log.String("orderType", string(orderTypeConfig.orderType)),
						log.String("subtype", string(subtypeConfig.subtype)),
					)
					registerError(errorsCount, nil, orderTypeConfig.orderType, subtypeConfig.subtype)
					continue
				}
				for _, block := range blockByType {
					var err error
					switch block := block.(type) {
					case *ui.Header:
						err = blocktests.TestHeaderBlock(block)
					case ui.Greeting:
						err = blocktests.TestGreetingBlock(block)
					case ui.UsefulBlock:
						err = blocktests.TestUsefulBlock(block)
					case ui.WeatherBlock:
						err = blocktests.TestWeatherBlock(block)
					case ui.CarouselBlock:
						err = blocktests.TestCarouselBlock(block)
					case ui.BeforeDepartureBlock:
						err = blocktests.TestBeforeDepartureBlock(block)
					case ui.FooterBlock:
						err = blocktests.TestFooterBlock(block)
					case ui.Disclaimers:
						err = blocktests.TestDisclaimersBlock(block)
					default:
						logger.Errorf("unknown block: %v", reflect.TypeOf(block))
					}
					if err != nil {
						if _, ok := errorsCache[err.Error()]; !ok {
							logger.Error(err.Error())
							errorsCache[err.Error()] = true
						}
						registerError(errorsCount, reflect.TypeOf(block), orderTypeConfig.orderType, subtypeConfig.subtype)
					}
				}
			}
		}
	}
	logger.Info(
		"Summary",
		log.Int("variants", variantsCount),
	)
	for k, v := range errorsCount {
		logger.Info(k, log.Int("errors", v))
	}
}

func registerError(errors map[string]int, blockType reflect.Type, orderType models.OrderType, subtype models.NotificationSubtype) {
	label := fmt.Sprintf("%v - %v - %v", blockType, subtype, orderType)
	current, ok := errors[label]
	if !ok {
		errors[label] = 1
	} else {
		errors[label] = current + 1
	}
}

type stationCodeSeeker struct {
	stationsRepository     *dicts.StationsRepository
	stationCodesRepository *dicts.StationCodesRepository
}

func newStationCodeSeeker(dictsRegistry *dicts.Registry) stationCodeSeeker {
	return stationCodeSeeker{
		stationsRepository:     dictsRegistry.GetStationsRepository(),
		stationCodesRepository: dictsRegistry.GetStationCodesRepository(),
	}
}

func (s *stationCodeSeeker) seekStationExpressCode(settlementID int) (string, error) {
	stations, ok := s.stationsRepository.GetBySettlementID(settlementID)
	if !ok || len(stations) == 0 {
		return "", fmt.Errorf("got no stations for the settlement")
	}
	for _, station := range stations {
		if station == nil {
			continue
		}
		stationExpressCode, ok := s.stationCodesRepository.SeekStationExpressCodeByID(station.Id)
		if !ok {
			continue
		}
		return stationExpressCode, nil
	}
	return "", fmt.Errorf("got no station express codes for the settlement")
}
