package main

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/dicts"
	"a.yandex-team.ru/travel/notifier/internal/extractors"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/processor"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/scheduling"
	"a.yandex-team.ru/travel/notifier/internal/service/rollout"
	"a.yandex-team.ru/travel/notifier/internal/service/scheduler"
)

const (
	mandatoryEmailSuffix = "@yandex-team.ru"
)

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &processor.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.NewDeploy(&processor.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	// for local debugging only
	isDebugRun := processor.Cfg.Debug.Email != "" && processor.Cfg.Debug.OrderID != ""

	defer func() {
		err = logger.L.Sync()
		if err != nil && !isDebugRun {
			fmt.Println("failed to close logger:", err)
		}
		ctxCancel()
	}()

	if !isDebugRun {
		logger.Error("please specify debug email amd orderID")
		return
	}

	if !strings.HasSuffix(processor.Cfg.Debug.Email, mandatoryEmailSuffix) {
		logger.Errorf("test email has to end with %s", mandatoryEmailSuffix)
		return
	}

	tvmClient, err := tvmtool.NewDeployClient()
	if err != nil {
		logger.Errorf("failed to create tvm client: %s", err)
	}

	pgClient, err := processor.BuildPGClient(logger)
	if err != nil {
		logger.Errorf("failed to create PG client: %s", err)
		return
	}

	recipientsRepository := database.NewRecipientsRepository(pgClient)
	ordersRepository := database.NewOrdersRepository(pgClient)
	notificationsRepository := processor.BuildNotificationsRepository(pgClient)
	senderClient := processor.BuildSenderClient(logger)
	ordersClient := processor.BuildOrdersClient(logger)
	if ordersClient == nil {
		logger.Error("failed to create a client to the orders app")
		return
	}

	dictsRegistry, err := dicts.NewRegistry(processor.Cfg.Dicts, logger)
	if err != nil {
		logger.Error("failed to create dicts registry", log.Error(err))
		return
	}
	notificationsBuilder := scheduling.NewNotificationsBuilder(
		scheduling.NotificationsBuilderConfig{
			IsTesting:                 processor.Cfg.IsTesting(),
			TestingEmails:             containers.SetOf(processor.Cfg.Pretrip.Testing.Emails...),
			AdhocSendingInterval:      processor.Cfg.Pretrip.Testing.AdhocSendingInterval,
			WeekBeforeSendingInterval: processor.Cfg.Pretrip.Testing.WeekBeforeSendingInterval,
			DayBeforeSendingInterval:  processor.Cfg.Pretrip.Testing.DayBeforeSendingInterval,
		},
	)
	notificationsScheduler := scheduler.NewService(logger, notificationsRepository, recipientsRepository)
	clock := clockwork.NewRealClock()
	unifiedAgentClient, _ := unifiedagent.NewGrpcClient(&unifiedagent.ClientConfig{Enabled: false}, logger, nil)

	settlementDataProvider := extractors.NewSettlementDataProvider(
		dictsRegistry.GetSettlementsRepository(),
		dictsRegistry.GetTimeZonesRepository(),
	)

	stationDataProvider := extractors.NewStationDataProvider(
		dictsRegistry.GetStationsRepository(),
		dictsRegistry.GetStationToSettlementRepository(),
		dictsRegistry.GetSettlementsRepository(),
		dictsRegistry.GetStationCodesRepository(),
	)

	pretripProcessor, err := processor.BuildPretripProcessor(
		logger,
		notificationsRepository,
		recipientsRepository,
		ordersRepository,
		senderClient,
		ordersClient,
		tvmClient,
		dictsRegistry,
		notificationsBuilder,
		notificationsScheduler,
		clock,
		rollout.NewService(rollout.Config{Percentage: 100}),
		unifiedAgentClient,
		settlementDataProvider,
		stationDataProvider,
	)
	if err != nil {
		logger.Error("failed to create pretrip processor", log.Error(err))
		return
	}

	notifications, err := notificationsRepository.GetForOrder(
		ctx,
		processor.Cfg.Debug.OrderID,
		time.Now().AddDate(-1, 0, 0),
		time.Now().AddDate(1, 0, 0),
		false,
	)
	if err != nil {
		logger.Error(
			"failed to get notifications for the specified order",
			log.Error(err),
		)
		return
	}
	if len(notifications) == 0 {
		logger.Info("no notifications has been found for the specified order")
		return
	}
	latest, ok := findLatest(
		notifications, func(notification models.Notification) bool {
			return processor.Cfg.Debug.NotificationSubtype == "" || string(notification.Subtype) == processor.Cfg.Debug.NotificationSubtype
		},
	)
	if !ok {
		latest, ok = findLatest(
			notifications, func(notification models.Notification) bool {
				return true
			},
		)
		// substitute subtype on the fly if there's no notification with the desired one in the database
		latest.Subtype = models.NotificationSubtype(processor.Cfg.Debug.NotificationSubtype)
	}
	if !ok {
		logger.Infof("no notifications for the specified order has the specified subtype %s", processor.Cfg.Debug.NotificationSubtype)
		return
	}
	if latest.Recipient.UnsubscribeHash == nil {
		localHashString := "hash"
		latest.Recipient.UnsubscribeHash = &localHashString
	}

	pretripProcessor.DebugSend(ctx, &latest, processor.Cfg.Debug.Email)
}

func findLatest(notifications []models.Notification, predicate func(models.Notification) bool) (models.Notification, bool) {
	if len(notifications) == 0 {
		return models.Notification{}, false
	}

	latest := notifications[0]
	found := false
	for _, n := range notifications {
		if predicate(n) {
			if !found {
				latest = n
			}
			found = true
		} else {
			continue
		}
		if n.UpdatedAt.After(latest.UpdatedAt) {
			latest = n
		}
	}
	return latest, found
}
